/*
 * Copyright (c) 2005, John Mettraux, OpenWFE.org
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * . Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.  
 * 
 * . Redistributions in binary form must reproduce the above copyright notice, 
 *   this list of conditions and the following disclaimer in the documentation 
 *   and/or other materials provided with the distribution.
 * 
 * . Neither the name of the "OpenWFE" nor the names of its contributors may be
 *   used to endorse or promote products derived from this software without
 *   specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: MailListener.java 2414 2006-03-10 22:15:03Z jmettraux $
 */

//
// MailListener.java
//
// jmettraux@openwfe.org
//
// generated with 
// jtmpl 1.0.04 20.11.2001 John Mettraux (jmettraux@openwfe.org)
//

package openwfe.org.engine.impl.listen;

import javax.mail.Flags;
import javax.mail.Store;
import javax.mail.Folder;
import javax.mail.Session;
import javax.mail.URLName;
import javax.mail.Message;

import openwfe.org.Utils;
import openwfe.org.MapUtils;
import openwfe.org.Application;
import openwfe.org.ServiceException;
import openwfe.org.ApplicationContext;
import openwfe.org.mail.MailUtils;
import openwfe.org.engine.listen.WorkItemListener;
import openwfe.org.engine.workitem.WorkItem;
import openwfe.org.engine.workitem.WorkItemCoder;
import openwfe.org.engine.impl.dispatch.SmtpDispatcher;


/**
 * A daemon that polls an email account (POP3 or IMAP) for incoming workitems
 * (email payload).
 *
 * <p><font size=2>CVS Info :
 * <br>$Author: jmettraux $
 * <br>$Date: 2006-03-10 23:15:03 +0100 (Fri, 10 Mar 2006) $
 * <br>$Id: MailListener.java 2414 2006-03-10 22:15:03Z jmettraux $ </font>
 *
 * @author john.mettraux@openwfe.org
 */
public class MailListener

    extends WorkItemListener

{

    private final static org.apache.log4j.Logger log = org.apache.log4j.Logger
        .getLogger(MailListener.class.getName());

    //
    // CONSTANTS (definitions)

    /**
     * The parameter 'accountUrl' tells this listener where it should
     * poll for incoming workitem emails.
     * It should be of the form : protocol://username:password@host:port
     */
    public final static String P_ACCOUNT_URL
        = "accountUrl";

    /**
     * For a POP3 account, this parameter 'folder' is not necessary and
     * defaults to 'INBOX', it thus takes all its importance for IMAP
     * accounts.
     */
    public final static String P_FOLDER
        = "folder";

    /**
     * Use the 'frequency' parameter to tell this listener how frequently
     * it should poll for new workitem emails.
     * The default time is '30s'.
     */
    public final static String P_FREQUENCY
        = "frequency";

    private final static String DEFAULT_FREQUENCY
        = "30s"; // every 30 seconds

    //
    // FIELDS

    private URLName account = null;
    private String folder = null;

    private java.util.TimerTask pollTask = null;

    //
    // CONSTRUCTORS

    public void init 
        (final String serviceName, 
         final ApplicationContext context, 
         final java.util.Map serviceParams)
    throws 
        ServiceException
    {
        // using the super init method is mandatory as it instatiates the
        // work item consumer that the listener will use.

        super.init(serviceName, context, serviceParams);

        //
        // email account

        String sAccount = MapUtils.getMandatoryString
            (serviceParams, P_ACCOUNT_URL);

        this.account = new URLName(sAccount);

        // 
        // folder

        this.folder = MapUtils.getAsString
            (serviceParams, P_FOLDER, "INBOX");

        if (sAccount.startsWith("pop")) this.folder = "INBOX";

        //
        // frequency

        long frequency = MapUtils
            .getAsTime(serviceParams, P_FREQUENCY, DEFAULT_FREQUENCY);

        //
        // start the service !

        this.pollTask = new java.util.TimerTask()
        {
             public void run ()
             {
                 poll();
             }
        };

        Application.getTimer().schedule(this.pollTask, 10, frequency);
    }

    //
    // METHODS

    /**
     * Takes care of stopping the polling.
     */
    public void stop ()
        throws ServiceException
    {
        super.stop();

        this.pollTask.cancel();

        log.info("'"+getName()+"' Stopped.");
    }

    /**
     * This method does the job of polling the POP/IMAP account for mails
     * containing workitems.
     */
    protected synchronized void poll ()
    {
        Session mailSession = null;
        Folder mailFolder = null;
        Store mailStore = null;

        try
        {
            mailSession = MailUtils.getMailSession(getParams());

            mailStore = mailSession.getStore(this.account);
            mailStore.connect();

            mailFolder = mailStore.getFolder(this.folder);
            mailFolder.open(Folder.READ_WRITE);

            //
            // play with messages

            final Message[] messages = mailFolder.getMessages();

            log.debug("poll() examining "+messages.length+" messages");

            for (int i=0; i<messages.length; i++)
            {
                final Message m = messages[i];

                log.debug("poll() message subject is >"+m.getSubject()+"<");

                try
                {
                    handleMessage(m);

                    //
                    // delete message from folder

                    m.setFlag(Flags.Flag.DELETED, true);
                }
                catch (final Exception e)
                {
                    log.warn
                        ("Failed to handle message >"+m.getSubject()+
                         "<. Skipped.");
                }
            }
        }
        catch (final Throwable t)
        {
            log.warn("poll() failure", t);
        }
        finally
        {
            try
            {
                mailFolder.close(true);
            }
            catch (final Throwable t)
            {
                // ignore
            }

            try
            {
                mailStore.close();
            }
            catch (final Throwable t)
            {
                // ignore
            }
        }
    }

    /**
     * Does the work of handling the message.
     */
    protected void handleMessage (final Message m)
        throws Exception
    {
        //
        // extract payload
        
        final String s = m.getContent().toString();

        final int n = s.indexOf("\n");

        final String coderName = s.substring(0, n).trim();
        final String sWorkItem = s.substring(n+1).trim();

        //log.debug("handleMessage() workitem is :\n"+s);

        final WorkItem wi = getCoder(coderName)
            .decode(sWorkItem, getContext(), getParams());

        getConsumer().use(wi);
    }

}
