/*
 * Copyright (c) 2001-2006, John Mettraux, OpenWFE.org
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * . Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.  
 * 
 * . Redistributions in binary form must reproduce the above copyright notice, 
 *   this list of conditions and the following disclaimer in the documentation 
 *   and/or other materials provided with the distribution.
 * 
 * . Neither the name of the "OpenWFE" nor the names of its contributors may be
 *   used to endorse or promote products derived from this software without
 *   specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: EngineWorkItemConsumer.java 2809 2006-06-13 07:47:34Z jmettraux $
 */

//
// EngineWorkItemConsumer.java
//
// jmettraux@openwfe.org
//
// generated with 
// jtmpl 1.0.04 20.11.2001 John Mettraux (jmettraux@openwfe.org)
//

package openwfe.org.engine.listen;

import openwfe.org.MapUtils;
import openwfe.org.OpenWfeException;
import openwfe.org.ServiceException;
import openwfe.org.ApplicationContext;
import openwfe.org.xml.XmlUtils;
import openwfe.org.engine.Definitions;
import openwfe.org.engine.launch.Launcher;
import openwfe.org.engine.expool.ExpressionPool;
import openwfe.org.engine.history.History;
import openwfe.org.engine.workitem.WorkItem;
import openwfe.org.engine.workitem.LaunchItem;
import openwfe.org.engine.workitem.InFlowWorkItem;
import openwfe.org.engine.workitem.Attribute;
import openwfe.org.engine.workitem.XmlAttribute;
import openwfe.org.engine.workitem.ListAttribute;
import openwfe.org.engine.workitem.StringAttribute;
import openwfe.org.engine.expressions.ReplyException;
import openwfe.org.engine.expressions.FlowExpressionId;


/**
 * This consumer simply takes care of routing incoming workitems to the
 * appropriate engine service.
 * Incoming workitems arrive through the listeners configured for the engine.
 *
 * <p><font size=2>CVS Info :
 * <br>$Author: jmettraux $
 * <br>$Date: 2006-06-13 09:47:34 +0200 (Tue, 13 Jun 2006) $
 * <br>$Id: EngineWorkItemConsumer.java 2809 2006-06-13 07:47:34Z jmettraux $ </font>
 *
 * @author john.mettraux@openwfe.org
 */
public class EngineWorkItemConsumer

    extends AbstractWorkItemConsumer

{

    private final static org.apache.log4j.Logger log = org.apache.log4j.Logger
        .getLogger(EngineWorkItemConsumer.class.getName());

    //
    // CONSTANTS and co

    /**
     * If this 'processDefinitionInField' parameter is set, the consumer
     * will take care of checking the named field to see if it contains
     * a process definition; if so, this 'embedded' definitionn will
     * be executed.
     */
    public final static String P_PROCESS_DEFINITION_IN_FIELD
        = "processDefinitionInField";

    /**
     * This param 'archiveProcessDefinitionInField' is only effective if
     * the param 'processDefinitionInField' is set; the value of this 
     * parameter is the name of the field that will hold the list of 'consumed'
     * process definitions. Else consumed process defintions are just 
     * discarded. Archiving process definitions may lead to 'fat' workitems,
     * but may be interesting for tracing purposes.
     */
    public final static String P_ARCHIVE_PROCESS_DEFINITION_IN_FIELD
        = "archiveProcessDefinitionInField";

    //
    // FIELDS

    private String processDefinitionField = null;
    private String archiveProcessDefinitionField = null;

    //
    // CONSTRUCTORS

    public void init 
        (final String serviceName, 
         final ApplicationContext context, 
         final java.util.Map serviceParams)
    throws 
        ServiceException
    {
        super.init(serviceName, context, serviceParams);

        // 'processDefinitionInField'

        this.processDefinitionField = MapUtils
            .getAsString(serviceParams, P_PROCESS_DEFINITION_IN_FIELD);

        if (this.processDefinitionField != null)
        {
            log.info
                ("init() processDefinitionField set to '"+
                 this.processDefinitionField+"'");
        }

        // 'archiveProcessDefinitionInField'

        this.archiveProcessDefinitionField = MapUtils
            .getAsString(serviceParams, P_ARCHIVE_PROCESS_DEFINITION_IN_FIELD);

        if (this.archiveProcessDefinitionField != null)
        {
            log.info
                ("init() archiveProcessDefinitionField set to '"+
                 this.archiveProcessDefinitionField+"'");
        }
    }

    //
    // METHODS

    /**
     * This method looks at the wi and either reply to the corresponding
     * ParticipantExpression either launch a new wf.
     */
    public Object use (final WorkItem wi)
        throws OpenWfeException
    {
        if (wi instanceof LaunchItem)
        {
            return getLauncher().launch((LaunchItem)wi, true);
        }

        if ( ! (wi instanceof InFlowWorkItem))
        {
            throw new OpenWfeException
                ("The engine cannot use a WorkItem of class "+
                 wi.getClass().getName());
        }

        final InFlowWorkItem ifwi = (InFlowWorkItem)wi;

        if (log.isDebugEnabled())
        {
            log.debug
                ("use() received workitem with id  "+
                 ifwi.getLastExpressionId());
            log.debug
                ("use() received workitem for participant '"+
                 ifwi.getParticipantName()+"'");
        }

        if (this.processDefinitionField != null)
            //
            // look to see if there is an embedded process definition
            // to 'consume'
        {
            final Attribute a = 
                ifwi.getAttributes().getField(this.processDefinitionField);

            if (a != null &&
                ((a instanceof StringAttribute) ||
                 (a instanceof XmlAttribute)))
            {
                archiveProcessDefinition(a, ifwi);

                getLauncher().launch
                    (ifwi,
                     ifwi.getId(),
                     Launcher.URL_FIELD_PREFIX+this.processDefinitionField,
                     true); // async is true

                return null;
            }
        }

        //
        // the normal case

        //reply((InFlowWorkItem)wi);

        (new Thread()
         {
             public void run ()
             {
                 try
                 {
                    reply((InFlowWorkItem)wi);
                 }
                 catch (final Throwable t)
                 {
                     log.error("use() problem while replying", t);
                 }
             }
         }).start();

        return null;
    }

    /**
     * If the 'archiveProcessDefinitionInField' parameter is set, 
     * this method will take care of storing the incoming process definition
     * in the 'archive' list field (attribute).
     */
    protected void archiveProcessDefinition 
        (final Attribute definition, final InFlowWorkItem wi)
    {
        if (this.archiveProcessDefinitionField == null)
            return;

        Attribute a = wi.getAttributes()
            .getField(this.archiveProcessDefinitionField);

        if (a != null && ( ! (a instanceof ListAttribute)))
        {
            log.error
                ("archiveProcessDefinition() "+
                 "field '"+this.archiveProcessDefinitionField+
                 "' is of class "+a.getClass().getName());
            return;
        }

        ListAttribute archive = (ListAttribute)a;

        if (archive == null) 
        {
            archive = new ListAttribute();

            wi.getAttributes()
                .setField(this.archiveProcessDefinitionField, archive);
        }

        archive.add(definition);

        /*
        if (log.isDebugEnabled())
        {
            log.debug
                ("archiveProcessDefinition() "+
                 "there's now "+archive.size()+
                 " defs in archive field '"+
                 this.archiveProcessDefinitionField+"'");
        }
        */
    }

    /**
     * Takes care of handling the workitem (reply) to the expression pool.
     */
    protected void reply (final InFlowWorkItem wi)
        throws ReplyException
    {
        //if (log.isDebugEnabled())
        //    log.debug("reply() to  "+wi.getLastExpressionId());

        final History h = Definitions.getHistory(getContext());

        if (h != null)
        {
            h.log
                (wi.getLastExpressionId(),
                 wi,
                 History.EVT_INCOMING,
                 wi.getParticipantName(),
                 ""); 
        }

        Definitions.getExpressionPool(getContext())
            .reply(wi.getLastExpressionId(), wi);
    }

    /**
     * Returns (if set) the name of the field who is supposed to hold
     * an embedded process definition.
     */
    public String getProcessDefinitionField ()
    {
        return this.processDefinitionField;
    }

    /**
     * Returns the name of the field that should hold the list of 
     * process definitions previously 'consumed on the fly'.
     * If this field is not set, embedded process definition are just
     * discarded after having been consumed.
     */
    public String getArchiveProcessDefinitionField ()
    {
        return this.archiveProcessDefinitionField;
    }

    /**
     * Returns the launcher as found in the current application context.
     */
    protected Launcher getLauncher ()
    {
        return Definitions.getLauncher(this.getContext());
    }

    //
    // METHODS from Service

    /**
     * Not much info to return...
     */
    public org.jdom.Element getStatus ()
    {
        final org.jdom.Element result = new org.jdom.Element(getName());

        result.addContent(XmlUtils.getClassElt(this));
        result.addContent(XmlUtils.getRevisionElt("$Id: EngineWorkItemConsumer.java 2809 2006-06-13 07:47:34Z jmettraux $"));

        return result;
    }

}
