/*
 * Copyright (c) 2006, Nicolas Modrzyk and John Mettraux, OpenWFE.org
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * . Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.  
 * 
 * . Redistributions in binary form must reproduce the above copyright notice, 
 *   this list of conditions and the following disclaimer in the documentation 
 *   and/or other materials provided with the distribution.
 * 
 * . Neither the name of the "OpenWFE" nor the names of its contributors may be
 *   used to endorse or promote products derived from this software without
 *   specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: ReflectionUtils.java 2673 2006-05-26 21:08:46Z jmettraux $
 */

//
// JCRBeanCoder.java
//
// Nicolas Modrzyk
// john.mettraux@openwfe.org
//

package openwfe.org.jcr.beancoder;

//import java.lang.reflect.Method;

import javax.jcr.PropertyType;

import openwfe.org.jcr.Node;
import openwfe.org.jcr.Item;
import openwfe.org.jcr.Value;
import openwfe.org.jcr.Property;
import openwfe.org.jcr.JcrProxy;
import openwfe.org.jcr.JcrException;
import openwfe.org.jcr.JcrValueUtils;

//import openwfe.org.misc.Base64;
import openwfe.org.misc.ByteUtils;
import openwfe.org.util.beancoder.BeanCoderUtils;
import openwfe.org.util.beancoder.AbstractBeanCoder;
import openwfe.org.util.beancoder.BeanCoderException;


/**
 * The first implementation of BeanCoder; will supplant xml.XmlCoder soon.
 *
 * <p><font size=2>CVS Info :
 * <br>$Author$
 * <br>$Id$ </font>
 *
 * @author Nicolas Modrzyk
 * @author john.mettraux@openwfe.org
 */
public class JcrBeanCoder

    extends AbstractBeanCoder

{

    private final static org.apache.log4j.Logger log = org.apache.log4j.Logger
        .getLogger(JcrBeanCoder.class.getName());

    //
    // CONSTANTS & co

    public final static String PN_SIZE
        = "size";
    public final static String PN_PVALUE
        = "pvalue";

    public final static String PN_CLASS
        = JcrBeanCoder.class.getName() + "__" + "instance_class";

    public final static String NN_ENTRY
        = "map-entry";
    //public final static String NN_VALUE
    //    = "value";

    public final static String C_NULL
        = "Null";
    public final static String V_NULL
        = "null";

    //
    // FIELDS

    private String nsp = "";

    private Item previousItem = null;
    private Item currentItem = null;

    private String currentItemName = null;

    private long currentLocalId = -1;

    private boolean override = true;
    private boolean overrideDone = false;

    //
    // CONSTRUCTORS

    public JcrBeanCoder 
        (final String ns, final Node startNode)
    {
        this(ns, startNode, null);
    }

    public JcrBeanCoder 
        (final String ns, final Node startNode, final String beanNodeName)
    {
        super();

        if (ns != null) this.nsp = ns + ":";

        this.currentItem = startNode;
        this.currentItemName = beanNodeName;
    }

    //
    // METHODS

    /**
     * This coder 'property' is set to true by default; it means that the
     * coder, when encoding, will remove any JCR branch with the same name
     * before encoding to it.
     */
    public void setOverrideNode (final boolean b)
    {
        this.override = b;
    }

    /**
     * Returns true if this code shall remove any node with the same name as
     * the node it will ouput beforhand.
     */
    public boolean getOverrideNode ()
    {
        return this.override;
    }

    protected Node currentNode ()
    {
        return (Node)this.currentItem;
    }

    protected Property currentProperty ()
    {
        return (Property)this.currentItem;
    }

    protected void beginElement ()
        throws BeanCoderException
    {
        if (this.currentItemName == null) 
            //
            // use current node (todo #1560001)
            //
            return;

        this.currentItem = this.currentNode().addNode(this.currentItemName);

        this.currentItemName = null;
            //
            // will provoke NPEs :-)

        //log.debug("beginElement() '"+this.currentItem.getName()+"'");
    }

    protected /*synchronized*/ long getUniqueLocalId ()
    {
        this.currentLocalId++;
        return this.currentLocalId;
    }

    //
    // METHODS from AbstractBeanCoder

    // DECODING

    protected int currentType ()
        throws BeanCoderException
    {
        if (this.currentItem instanceof Property)
            return T_PRIMITIVE;

        final String className = currentClassName();

        if (className == null)
        {
            if (this.currentNode().hasProperty(this.nsp + PN_SIZE))
                return T_ARRAY;

            throw new JcrException
                ("property 'class' is missing "+
                 "("+this.currentItem.getPath()+")");
        }

        if (className.equals(C_NULL))
        {
            return T_NULL;
        }

        final Class c = currentClass();

        //if (byte[].class == c)
        //    return T_BYTE_ARRAY;

        if (java.util.Map.class.isAssignableFrom(c))
            return T_MAP;

        if (java.util.Collection.class.isAssignableFrom(c))
            return T_COLLECTION;

        return T_BEAN;
    }

    protected Class currentPrimitiveClass ()
        throws BeanCoderException
    {
        Value v = null;
        String s = null;

        v = this.currentProperty().getValue();

        if (v.getType() == PropertyType.STRING) return String.class;
        if (v.getType() == PropertyType.BOOLEAN) return Boolean.class;
        if (v.getType() == PropertyType.DATE) return java.util.Date.class;
        if (v.getType() == PropertyType.BINARY) return byte[].class;

        //s = v.getString();
        s = v.toString();

        if (v.getType() == PropertyType.DOUBLE)
        {
            if (s.endsWith("f")) return Float.class;
            return Double.class;
        }
        if (v.getType() == PropertyType.LONG)
        {
            if (s.endsWith("l")) return Long.class;
            return Integer.class;
        }

        throw new JcrException
            ("cannot handle property of type '"+
             PropertyType.nameFromValue(v.getType())+"'");
    }

    protected String currentClassName ()
        throws BeanCoderException
    {
        if ( ! this.currentItem.isNode())
            return currentPrimitiveClass().getName();

        if ( ! this.currentNode().hasProperty(this.nsp + PN_CLASS))
            return null;

        return this.currentNode()
            .getProperty(this.nsp + PN_CLASS).getString();
    }

    protected Class currentClass ()
        throws BeanCoderException
    {
        if ( ! this.currentItem.isNode())
            return currentPrimitiveClass();

        String className = null;
        try
        {
            className = this.currentNode()
                .getProperty(this.nsp + PN_CLASS).getString();

            return Class.forName(className);
        }
        catch (final JcrException e)
        {
            throw e;
        }
        catch (final Throwable t)
        {
            throw new BeanCoderException
                ("failed to find class named '"+className+"'", t);
        }
    }

    protected int subElementCount ()
        throws BeanCoderException
    {
        return (int)this.currentNode()
            .getProperty(this.nsp + PN_SIZE).getLong();
    }

    protected java.util.Iterator subElementIterator ()
        throws BeanCoderException
    {
        final int type = this.currentType();

        if (type == T_ARRAY || type == T_COLLECTION)
            return new IndexedIterator(this.currentNode());

        return new ItemIterator(this.currentNode());
    }

    protected String currentFieldName ()
        throws BeanCoderException
    {
        return this.currentItem.getName();
    }

    protected String currentText ()
        throws BeanCoderException
    {
        //if (this.currentProperty().getDefinition().isMultiple())
        //    text = this.currentProperty().getValues()[1].getString();
        //else
        return this.currentProperty().getString();
    }

    /**
     * Overriding decodeBean() to intercept decode request for
     * primitive encoded as beans (it occurs for certains map keys).
     */
    protected Object decodeBean ()
        throws BeanCoderException
    {
        if (this.currentNode().hasProperty(this.nsp + PN_PVALUE))
        {
            return BeanCoderUtils.newPrimitive
                (currentClass(),
                 this.currentNode().getProperty
                     (this.nsp + PN_PVALUE).getString());
        }

        return super.decodeBean();
    }

    protected Object decodeFieldValue ()
        throws BeanCoderException
    {
        if ( ! this.currentItem.isNode()) return decodePrimitive();

        return decode();
    }

    /**
     * Overriding the decodePrimitive() method of the parent class
     * to take care of binary values.
     */
    protected Object decodePrimitive ()
        throws BeanCoderException
    {
        final Class primitiveClass = currentClass();

        if (primitiveClass != byte[].class) 
            return super.decodePrimitive();

        return decodeByteArray();
    }

    /**
     * Overriding decodeMapEntry() to take care of the JCR properties
     * or node structuring...
     */
    protected Object[] decodeMapEntry ()
        throws BeanCoderException
    {
        //log.debug
        //    ("decodeMapEntry()");
        //log.debug
        //    ("decodeMapEntry() considering item '"+
        //     this.currentItem.getName()+"'");

        if ( ! this.currentItem.getName().startsWith(this.nsp))
            return null;

        if (this.currentItem.getName().equals(this.nsp + PN_CLASS))
            return null;

        Object key = null;
        Object value = null;

        if (this.currentItem.isNode())
            //
            // node
        {
            final Node nk = this.currentNode();

            final Node nv = nk.getNode(nk.getName());

            //this.currentItem = nk;

            key = decode();

            // then value

            this.currentItem = nv;

            value = decode();
        }
        else
            //
            // it's property
        {
            key = 
                this.currentProperty().getName().substring(this.nsp.length());
            value = 
                this.currentProperty().getValue().getValue();

            if (this.currentProperty().getValue().getType() == 
                    PropertyType.NAME &&
                value.equals(V_NULL))
            {
                value = null;
            }
        }

        //log.debug("decodeMapEntry() key is '"+key+"'");
        //log.debug("decodeMapEntry() val is '"+value+"'");

        return new Object[] { key, value };
    }

    protected byte[] decodeByteArray ()
        throws BeanCoderException
    {
        final java.io.ByteArrayOutputStream baos = 
            new java.io.ByteArrayOutputStream();

        final java.io.InputStream is = this.currentProperty()
            .getValue().getStream();

        try
        {
            ByteUtils.copy(is, baos, 1024);
        }
        catch (final Throwable t)
        {
            throw new BeanCoderException
                ("Failed to decode byte array for '"+this.currentItemName+
                 "' out of stream", t);
        }

        return baos.toByteArray();
    }

    // ENCODING

    protected void encodeByteArray (final byte[] array)
        throws BeanCoderException
    {
        this.currentNode().setProperty
            (this.currentItemName,
             new java.io.ByteArrayInputStream(array));
    }

    /**
     * Overriding encode() to remove potential previous nodes with the
     * same name ('override'). Only occurs at the very beginning of 
     * the encoding work.
     */
    public void encode (final Object o)
        throws BeanCoderException
    {
        if (this.override && ( ! this.overrideDone))
        {
            final java.util.Iterator it = this.currentNode().getNodes();
            while (it.hasNext())
            {
                final Node n = (Node)it.next();

                if (n.getName().equals(this.currentItemName)) n.remove();
            }

            this.overrideDone = true;
        }

        super.encode(o);
    }

    protected void encodeEntry (final int index, final Object entry)
        throws BeanCoderException
    {
        this.currentItemName = this.nsp + index;
        encode(entry);
    }

    protected void endElement ()
        throws BeanCoderException
    {
        this.previousItem = this.currentItem;
        this.currentItem = this.currentItem.getParent();

        //log.debug("endElement()");
        //log.debug("endElement() prev  '"+this.previousItem.getName()+"'");
        //log.debug("endElement() curr  '"+this.currentItem.getName()+"'");
    }

    protected void encodeNull ()
        throws BeanCoderException
    {
        beginElement();

        this.currentNode()
            .setProperty(this.nsp + PN_CLASS, C_NULL, PropertyType.NAME);
    }

    protected void encodePrimitive (final Object bean)
        throws BeanCoderException
    {
        this.currentNode().setProperty
            (this.currentItemName, 
             bean.toString(), 
             JcrValueUtils.getJcrType(bean));
    }

    protected void encodeField (final String fieldName, final Object value)
        throws BeanCoderException
    {
        //log.debug("encodeField() '"+fieldName+"'");

        final Node beanNode = this.currentNode();

        this.currentItemName = fieldName;
        encode(value);

        //if (this.currentItem != beanNode)
        //    endElement();
        
        if (this.currentItem != beanNode)
        {
            this.previousItem = this.currentItem;
            this.currentItem = beanNode;
        }
        
    }

    protected void beginBean (final Object bean)
        throws BeanCoderException
    {
        beginWithClass(bean);

        if (BeanCoderUtils.isPrimitive(bean))
            //
            // this may happen, for map entry keys that are primitive but
            // whose value is not a primitive
        {
            this.currentNode()
                .setProperty(this.nsp + PN_PVALUE, bean.toString());
        }
    }

    protected void beginArray (int length)
        throws BeanCoderException
    {
        beginElement();

        this.currentNode().setProperty
            (this.nsp + PN_SIZE, ""+length, PropertyType.LONG);
    }

    protected void beginWithClass (final Object bean)
        throws BeanCoderException
    {
        beginElement();

        this.currentNode().setProperty
            (this.nsp + PN_CLASS, bean.getClass().getName());
    }

    protected void beginMap (final java.util.Map map)
        throws BeanCoderException
    {
        beginWithClass(map);
    }

    /**
     * Returns true if the given instance, when turned into a String
     * doesn't contain any chars banned from JCR node/property name.
     */
    protected boolean isSafeForAnItemName (final Object o)
    {
        if ( ! (o instanceof String)) return false;

        // TODO : add checks here

        return true;
    }

    protected Object[] asPropertyMapEntry 
        (final Object key, final Object val)
    {
        if (isSafeForAnItemName(key) &&
            (val == null || BeanCoderUtils.isPrimitive(val)))
        {
            return new Object[] { key, val };
        }

        return null;
    }

    protected void encodeMapEntry (final java.util.Map.Entry e)
        throws BeanCoderException
    {
        //log.debug("encodeMapEntry() key '"+e.getKey()+"'");

        Object key = e.getKey();
        Object val = e.getValue();

        //if (isSafeForAnItemName(key) &&
        //    (val == null ||
        //     BeanCoderUtils.isPrimitive(val)))

        final Object[] pe = asPropertyMapEntry(key, val);

        if (pe != null)
        {
            key = pe[0];
            val = pe[1];

            int tVal = PropertyType.NAME;
            String sVal = V_NULL;

            if (val != null)
            {
                tVal = JcrValueUtils.getJcrType(val);
                sVal = val.toString();
            }

            this.currentNode().setProperty(key.toString(), sVal, tVal);

            return;
        }

        final Node mapNode = this.currentNode();

        final String keyNodeName =
            this.nsp + NN_ENTRY + "_" + getUniqueLocalId();

        // -> key

        this.currentItemName = keyNodeName;

        encodeBean(key);

        // -> key -> value

        this.currentItem = this.previousItem;

        this.currentItemName = keyNodeName;

        encode(val);
        
        // done.

        this.currentItem = mapNode;
    }

    protected void beginMapEntry ()
        throws BeanCoderException
    {
        // not used in this implementation, 
        // as encodeMapEntry got overriden.
    }

    protected void beginCollection (final java.util.Collection col)
        throws BeanCoderException
    {
        beginWithClass(col);
    }

    //
    // INNER CLASSES

    /**
     * Used when decoding Arrays, Collections; it positions 
     * currentItem on the array/collection element to consider.
     */
    protected class IndexedIterator
        implements java.util.Iterator
    {
        private Node node = null;
        private int position = 0;

        public IndexedIterator (final Node n)
        {
            this.node = n;
        }

        public boolean hasNext ()
        {
            final String sNext = JcrBeanCoder.this.nsp + position;

            try
            {
                if (this.node.hasProperty(sNext)) return true;
                if (this.node.hasNode(sNext)) return true;
            }
            catch (final JcrException e)
            {
                throw new RuntimeException
                    ("failed to determine if there's a next item", e);
            }

            return false;
        }

        public Object next ()
        {
            final String sNext = JcrBeanCoder.this.nsp + position;

            this.position++;

            try
            {
                if (this.node.hasProperty(sNext))
                {
                    JcrBeanCoder.this.currentItem = 
                        this.node.getProperty(sNext);
                }
                else
                {
                    JcrBeanCoder.this.currentItem = 
                        this.node.getNode(sNext);
                }
            }
            catch (final JcrException e)
            {
                throw new RuntimeException
                    ("failed to reach next item", e);
            }

            return null;
        }

        public void remove ()
        {
            // no need for an implementation of this method.
        }
    }

    protected class ItemIterator
        implements java.util.Iterator
    {
        private Node node = null;
        private java.util.Iterator pit = null;
        private java.util.Iterator nit = null;

        public ItemIterator (final Node n)
            throws JcrException
        {
            this.node = n;

            this.pit = n.getProperties();
            this.nit = n.getNodes();
        }

        public boolean hasNext ()
        {
            if (pit.hasNext()) return true;
            return nit.hasNext();
        }

        public Object next ()
        {
            Item next = null;

            if (this.pit.hasNext()) 
                next = (Item)this.pit.next();
            else
                next = (Item)this.nit.next();

            JcrBeanCoder.this.currentItem = next;

            return null;
        }

        public void remove ()
        {
            // no need for an implementation of this method.
        }
    }

    //
    // STATIC METHODS

    /**
     * A convenience method for encoding a bean to a JCR repository.
     * No namespace will be used.
     */
    public static void encode 
        (final javax.jcr.Node targetNode,
         final String beanNodeName,
         final Object bean)
    throws
        BeanCoderException
    {
        encode(null, targetNode, beanNodeName, bean);
    }

    /**
     * A convenience method for encoding a bean to a JCR repository.
     */
    public static void encode 
        (final String namespace, 
         final javax.jcr.Node targetNode,
         final String beanNodeName,
         final Object bean)
    throws
        BeanCoderException
    {
        final Node n = (Node)JcrProxy.wrap(targetNode);

        final JcrBeanCoder jbc = 
            new JcrBeanCoder(namespace, n, beanNodeName);

        jbc.encode(bean);
    }

    /**
     * A convenience method for decoding a bean from a JCR node.
     * A null namespace is used.
     */
    public static Object decode
        (final javax.jcr.Node beanNode)
    throws
        BeanCoderException
    {
        return decode(null, beanNode);
    }

    /**
     * A convenience method for decoding a bean from a JCR node.
     */
    public static Object decode
        (final String namespace,
         final javax.jcr.Node beanNode)
    throws
        BeanCoderException
    {
        final Node n = (Node)JcrProxy.wrap(beanNode);

        final JcrBeanCoder jbc = new JcrBeanCoder(namespace, n);

        return jbc.decode();
    }

}
