/*
 * Copyright (c) 2005-2006, John Mettraux, OpenWFE.org
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * . Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.  
 * 
 * . Redistributions in binary form must reproduce the above copyright notice, 
 *   this list of conditions and the following disclaimer in the documentation 
 *   and/or other materials provided with the distribution.
 * 
 * . Neither the name of the "OpenWFE" nor the names of its contributors may be
 *   used to endorse or promote products derived from this software without
 *   specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: WebflowServlet.java 3230 2006-09-05 04:17:50Z jmettraux $
 */

//
// WebflowServlet.java
//
// jmettraux@openwfe.org
//
// generated with 
// jtmpl 1.1.00 16.08.2003 John Mettraux (jmettraux@openwfe.org)
//

package openwfe.org.webflow;

import java.io.IOException;
import java.net.URL;
import java.net.URLClassLoader;
import javax.servlet.ServletConfig;
import javax.servlet.ServletException;
import javax.servlet.http.HttpSession;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import openwfe.org.ServiceException;
import openwfe.org.ApplicationContext;
import openwfe.org.misc.WebUtils;
import openwfe.org.engine.expool.ExpressionPool;
import openwfe.org.engine.launch.LaunchException;
import openwfe.org.engine.workitem.WorkItem;
import openwfe.org.engine.workitem.LaunchItem;
import openwfe.org.engine.workitem.CancelItem;
import openwfe.org.engine.workitem.InFlowWorkItem;
import openwfe.org.engine.workitem.AttributeUtils;
import openwfe.org.engine.dispatch.DispatchingException;
import openwfe.org.engine.expressions.ReplyException;
import openwfe.org.engine.expressions.FlowExpressionId;
import openwfe.org.engine.participants.ParticipantMap;
import openwfe.org.engine.impl.history.TextHistory;
import openwfe.org.embed.engine.Engine;
import openwfe.org.embed.engine.EmbeddedParticipant;
import openwfe.org.embed.impl.engine.InMemoryEngine;


/**
 * OpenWFE webflow control servlet
 *
 * <p><font size=2>CVS Info :
 * <br>$Author: jmettraux $
 * <br>$Id: WebflowServlet.java 3230 2006-09-05 04:17:50Z jmettraux $ </font>
 *
 * @author jmettraux@openwfe.org
 */
public class WebflowServlet

    extends javax.servlet.http.HttpServlet

    implements EmbeddedParticipant

{

    //
    // CONSTANTS & co

    public final static String F_HTTP_CONTEXT
        = "__http_context__";
    public final static String F_HTTP_SESSION_ID
        = "__http_session_id__";

    public final static String IP_START_PAGE
        = "startPage";

    public final static String IP_END_PAGE
        = "endPage";

    public final static String IP_HANDLER_PATH
        = "handlerPath";

    public final static String IP_FLOW_PATH
        = "flowPath";

    public final static String IP_HISTORY_PATH
        = "historyPath";

    public final static String P_WF_ACTION
        = "wfaction";

    public final static String P_FLOW
        = "flow";

    public final static String A_START
        = "start";

    public final static String WFSEQ
        = "wfseq";

    public final static String DEFAULT_HANDLER_PATH
        = "WEB-INF/handlers/";

    public final static String DEFAULT_FLOW_PATH
        = "WEB-INF/flows/";

    //
    // FIELDS

    private Engine flowEngine = null;

    //private ParticipantMap participantMap = null;
    private java.util.Map params = null;

    private java.net.URL handlerPath = null;
    private String flowPath = null;

    //
    // CONSTRUCTORS

    public WebflowServlet ()
    {
        super();

        try
        {
            this.flowEngine = new InMemoryEngine();

            //
            // setting 'async launch' to 'false'
            // async launches do break the webflow (putting it into an
            // inconsistent state)

            this.flowEngine.getContext()
                .put(ExpressionPool.P_ASYNC_LAUNCH, Boolean.FALSE);

            //
            // registering self as the unique participant

            this.flowEngine.registerParticipant(this);
        }
        catch (final ServiceException e)
        {
            log("build failure", e);
        }
    }

    //
    // METHODS from EmbeddedParticipant

    public void setEngine (final Engine e)
    {
        // nothing to do, the engine is already known...
    }

    /**
     * One of EmbeddedParticipant's method, in this implementation, it
     * determines if the workitem should be routed to a page (for display)
     * or to a handler (for manipulation).
     */
    public void consume (final WorkItem wi)
        throws Exception
    {
        final InFlowWorkItem workitem = (InFlowWorkItem)wi;

        if (workitem.getParticipantName().startsWith("/"))
            forwardToPage(workitem);
        else
            forwardToHandler(workitem);
    }

    public void cancel (final CancelItem ci)
    {
        // never used...
    }

    /**
     * This method is called for participant whose names begin with a /
     * (forward slash), they are pages and the webflow is forwarded to
     * them from here.
     * This method is 'protected' in order to let extending classes the 
     * opportunity to override it.
     */
    protected void forwardToPage (final InFlowWorkItem workitem)
        throws ServletException, IOException
    {
        log("forwardToPage()");

        final HttpContextAttribute hca = (HttpContextAttribute)workitem
            .getAttributes().get(F_HTTP_CONTEXT);

        WebflowUtils.putWorkitem(hca.getRequest(), workitem);

        WebUtils.forward
            (hca.getRequest(), 
             hca.getResponse(), 
             workitem.getParticipantName());
    }

    /**
     * Workitem targetting participants whose name doesn't begin with a
     * forward slash ("/") are treated here, there are usually forwarded
     * to a Handler implementation.
     * This method is 'protected' in order to let extending classes the 
     * opportunity to override it.
     */
    protected void forwardToHandler (final InFlowWorkItem workitem)
        throws Exception
    {
        log
            ("forwardToHandler() "+
             "loading handler >"+workitem.getParticipantName()+"<");

        //
        // reloads the path each time so time you don't have to restart
        // the webserver at each handler recompilation.
        //

        Class handlerClass = null;
        try
        {
            final ClassLoader loader = 
                new URLClassLoader(new URL[] { this.handlerPath });

            handlerClass = loader.loadClass(workitem.getParticipantName());
        }
        catch (final Throwable t)
        {
            log("forwardToHandler() did not found handler in handlerPath");
        }

        if (handlerClass == null)
            //
            // now looking in regular classpath
            handlerClass = Class.forName(workitem.getParticipantName());

        final Handler handler = (Handler)handlerClass.newInstance();

        final HttpContextAttribute hca = (HttpContextAttribute)workitem
            .getAttributes().get(F_HTTP_CONTEXT);

        final boolean shouldReplyToEngine = handler.handle(workitem);

        if (shouldReplyToEngine)
            this.flowEngine.reply(workitem);
    }

    //
    // METHODS from Participant

    /**
     * Calling this method will have no effect.
     */
    public void setRegex (final String s)
    {
        // ignore
    }

    /**
     * This method is called by the embedded engine. It will trigger
     * the consume() method.
     */
    public Object dispatch 
        (final ApplicationContext context, final WorkItem wi)
    throws 
        DispatchingException
    {
        if (wi instanceof CancelItem)
        {
            cancel((CancelItem)wi);
            return null;
        }

        //
        // Making sure to copy the workitem :
        // else problems may arise (the engine modifiying the workitem
        // while the participant holds it).
        // We're in the same JVM after all...
        //

        try
        {
            final InFlowWorkItem workitem = (InFlowWorkItem)wi;

            consume((InFlowWorkItem)workitem.clone());
            return null;
        }
        catch (final Throwable t)
        {
            log("dispatch() failure", t);

            throw new DispatchingException("dispatch() failure", t);
        }
    }

    public java.util.Map getParams ()
    {
        return this.params;
    }

    /**
     * The control servlet returns ".*" which matches any participant name,
     * so any participant lookup by the engine is intercepted by this
     * control servlet.
     */
    public String getRegex ()
    {
        //return this.regex;
        
        return ".*";
    }

    /**
     * The init method to implement as a Participant extension.
     */
    public void init 
        (//final ParticipantMap pMap,
         final String regex,
         final java.util.Map params)
    {
        //this.participantMap = pMap;
        //this.regex = regex;
        this.params = params;
    }

    //
    // METHODS from HttpServlet

    /**
     * The init from HttpServlet (overriden to load the flowPath and the
     * handlerPath from the init parameters).
     */
    public void init (final ServletConfig config)
        throws ServletException
    {
        super.init(config);

        //
        // flow path

        this.flowPath = getInitParameter(IP_FLOW_PATH);

        if (this.flowPath == null)
            this.flowPath = DEFAULT_FLOW_PATH;

        log("init() flowPath is "+this.flowPath);

        //
        // handler path

        String s = getInitParameter(IP_HANDLER_PATH);

        if (s == null) s = DEFAULT_HANDLER_PATH;

        s = config.getServletContext().getRealPath(s);

        try
        {
            this.handlerPath = (new java.io.File(s)).toURL();
        }
        catch (final java.net.MalformedURLException e)
        {
            throw new ServletException
                ("Failed to determine 'handlerPath' from "+s, e);
        }

        log("init() handlerPath : "+this.handlerPath);

        //
        // adding a TextHistory service to the embedded engine

        String historyPath = getInitParameter(IP_HISTORY_PATH);

        if (historyPath != null)
        {
            historyPath = config.getServletContext().getRealPath(historyPath);

            final java.util.Map hParams = new java.util.HashMap(1);

            hParams.put(TextHistory.P_FILE_NAME, historyPath);

            final TextHistory history = new TextHistory();

            log("init() historyPath: "+historyPath);
            
            try
            {
                history.init
                    (openwfe.org.engine.Definitions.S_HISTORY, 
                     this.flowEngine.getContext(), 
                     hParams);

                this.flowEngine.getContext().add(history);
            }
            catch (final Exception e)
            {
                log("init() failed to init TextHistory, doing without it.", e);
            }
        }
    }

    /**
     * This is the classical service() method of a servlet.
     */
    public void service
        (final HttpServletRequest req, final HttpServletResponse res)
    throws 
        ServletException, IOException
    {
        final String action = req.getParameter(P_WF_ACTION);

        final InFlowWorkItem wi = WebflowUtils.recomposeWorkitem(req);

        try
        {
            if (wi == null)
            {
                if (A_START.equals(action))
                {
                    start(req, res);
                    return;
                }

                WebUtils.forward(req, res, getInitParameter(IP_START_PAGE));
                return;
            }

            int reqSeq = getSeq(req);
            int sesReq = getSeq(req.getSession(false));

            log("service() reqSeq : "+reqSeq+"  sesReq : "+sesReq);

            if (reqSeq <= sesReq)
                //
                // show same page
            {
                consume(wi);
                return;
            }

            //
            // proceed 

            incSeq(req.getSession(false));

            wi.getAttributes().put
                (F_HTTP_CONTEXT, new HttpContextAttribute(req, res));

            this.flowEngine.reply(wi);

            //
            // The engine will reply : the consume() method will get
            // triggered, with its forward.
            // If not the response wil not get committed and the
            // forward at the end of this method will be used.
            //
        }
        catch (final Throwable t)
        {
            log("service() error", t);

            WebflowUtils.printPlainError(res, t);
        }

        if ( ! res.isCommitted())
            //
            // engine did not reply (the consume method did not use
            // the response)
            //
        {
            log("service() res is not committed (forwarding to end page)");

            //
            // remove workitem  from session

            final HttpSession session = req.getSession(false);

            if (session != null)
                WebflowUtils.removeWorkitem(session);

            //
            // forward to end page

            WebUtils.forward(req, res, getInitParameter(IP_END_PAGE));
        }
    }

    /**
     * starts a flow
     */
    protected void start
        (final HttpServletRequest req, final HttpServletResponse res)
    throws 
        LaunchException
    {
        final HttpSession oldSession = req.getSession(false);
        if (oldSession != null) oldSession.invalidate();

        final HttpSession session = req.getSession(true);

        log("start() new session is "+session.getId());

        incSeq(session);

        String flow = req.getParameter(P_FLOW);

        log("start() flow is >"+flow+"<");

        //flow = getServletContext().getRealPath("/")+"/WEB-INF/flows/"+flow;
        flow = getServletContext().getRealPath("/")+"/"+this.flowPath+flow;

        log("start() flow is >"+flow+"<");

        final LaunchItem li = new LaunchItem();

        li.setWorkflowDefinitionUrl(flow);

        li.getAttributes().put
            (F_HTTP_CONTEXT, new HttpContextAttribute(req, res));
        li.getAttributes().puts
            (F_HTTP_SESSION_ID, session.getId());

        //
        // feed the 'start' request parameters into the launchitem

        final java.util.Enumeration en = req.getParameterNames();
        while (en.hasMoreElements())
        {
            final String paramName = (String)en.nextElement();
            final String[] values = req.getParameterValues(paramName);

            if (values.length == 1)
            {
                li.getAttributes()
                    .puts(paramName, values[0]);
            }
            else
            {
                li.getAttributes()
                    .put(paramName, AttributeUtils.java2owfe(values));
            }
        }

        final FlowExpressionId fei = this.flowEngine.launch(li, false);

        log("start() flowid is "+fei.getWorkflowInstanceId());
    }

    private static int getSeq (final HttpServletRequest req)
    {
        final String s = req.getParameter(WFSEQ);
        
        if (s == null) return -1;

        return Integer.parseInt(s);
    }

    private static int getSeq (final HttpSession session)
    {
        if (session == null) return -1;

        final Integer i = (Integer)session.getAttribute(WFSEQ);

        if (i == null) return -1;

        return i.intValue();
    }

    /**
     * Use this method to retrieve the sequence number from the session.
     * This is especially useful from a page.
     */
    public static int sequence (final HttpSession session)
    {
        final int seq = getSeq(session);

        if (seq < 0) return seq;

        return seq + 1;
    }

    private static void incSeq (final HttpSession session)
    {
        if (session == null) return;

        final int currentSeq = getSeq(session);

        session.setAttribute(WFSEQ, new Integer(currentSeq+1));
    }

    /**
     * Returns "control?wfseq=xxx" where xxx is produced by the
     * sequence() method.
     * Of course, the resulting string is passed through 
     * HttpServletResponse.encodeURL().
     *
     * @see #sequence
     */
    public static String getActionURL 
        (final HttpServletResponse res, final HttpSession ses)
    {
        return res.encodeURL("control?"+WFSEQ+"="+sequence(ses));
    }

}
