package org.opoo.ootp.client;

import java.util.Arrays;
import java.util.List;

public interface ExsClient {
    /**
     * 默认上限。
     */
    int DEFAULT_LIMIT = 100;

    /**
     * 发送消息。
     * @param type 消息业务类型，用于消息处理方进行标识
     * @param to 接收方
     * @param contentType 消息内容类型
     * @param body 消息体正文
     * @throws OotpException 通常是 Http 请求异常，或者服务器返回错误消息
     */
    void send(String type, String to, String contentType, String body) throws OotpException;

    /**
     * 发送消息。
     * @param type 消息业务类型，用于消息处理方进行标识
     * @param to 接收方
     * @param contentType 消息内容类型
     * @param body 消息体正文
     * @throws OotpException 通常是 Http 请求异常，或者服务器返回错误消息
     */
    void send(String type, String to, String contentType, byte[] body) throws OotpException;

    /**
     * 轮询消息。
     * @param limit 每次轮询最大记录数，默认 100
     * @param autoAck 是否自动进行消息确认，即轮询时就发送消息回执，默认 false
     * @param full 是否返回完整的消息记录，默认 false
     * @return 消息集合，可能为空，不为 null
     * @throws OotpException 通常是 Http 请求异常，或者服务器返回错误消息
     */
    List<ExsMessage> poll(int limit, boolean autoAck, boolean full) throws OotpException;

    /**
     * 使用默认参数轮询消息。
     * @return 消息集合，可能为空，不为 null
     * @throws OotpException 通常是 Http 请求异常，或者服务器返回错误消息
     */
    default List<ExsMessage> poll() throws OotpException {
        return poll(DEFAULT_LIMIT, false, false);
    }

    /**
     * 获取消息体。
     * @param id 消息ID
     * @return 消息体，仅含消息正文
     * @throws OotpException 通常是 Http 请求异常，或者服务器返回错误消息
     */
    default String getMessageBody(String id) throws OotpException {
        return getMessageContent(id).getBody();
    }

    /**
     * 获取消息体。
     * @param id 消息ID
     * @return 消息体，含消息的 ContentType 信息
     * @throws OotpException 通常是 Http 请求异常，或者服务器返回错误消息
     */
    ExsMessageContent getMessageContent(String id) throws OotpException;

    /**
     * 确认消息。
     * @param ids 要确认的消息数组，不能为空
     * @throws OotpException 通常是 Http 请求异常，或者服务器返回错误消息
     */
    default void ack(String... ids) throws OotpException {
        ack(Arrays.asList(ids));
    }

    /**
     * 确认消息。
     * @param ids 要确认的消息集合，不能为空
     * @throws OotpException 通常是 Http 请求异常，或者服务器返回错误消息
     */
    void ack(List<String> ids) throws OotpException;

}
