package org.opoo.ootp.client;

import org.springframework.util.StreamUtils;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;

public interface OotpClient {
    String STORAGE_DEFAULT = "default";
    String STORAGE_FS = "fs";

    /**
     * 获取数据交换客户端。
     * @return 数据交换客户端。
     */
    ExsClient getExsClient();

    /**
     * 上传文件到文件服务文件库。
     *
     * @param repo 目标文件库
     * @param contentType 文件内容类型
     * @param file 文件
     * @return 上传后返回的文件ID
     * @throws IOException 文件处理异常
     * @throws OotpException 通常是 Http 请求异常，或者服务器返回错误消息
     */
    default String uploadToFsStorage(String repo, String contentType, File file) throws IOException, OotpException {
        return upload(STORAGE_FS, repo, contentType, file);
    }

    /**
     * 上传文件到文件服务文件库。
     *
     * @param repo 目标文件库
     * @param contentType 文件内容类型
     * @param file 文件
     * @param pathInfo 文件上传的后置处理
     * @return 上传后返回的文件ID
     * @throws IOException 文件处理异常
     * @throws OotpException 通常是 Http 请求异常，或者服务器返回错误消息
     */
    default String uploadToFsStorage(String repo, String contentType, File file, String pathInfo) throws IOException, OotpException {
        return upload(STORAGE_FS, repo, contentType, file, pathInfo);
    }

    /**
     * 上传文件到文件服务文件库。
     *
     * @param repo 目标文件库
     * @param contentType 文件内容类型
     * @param file 文件内容
     * @param fileName 文件名
     * @param fileSize 文件大小，未知大小传 -1
     * @param lastModified 文件最后更新时间
     * @return 上传后返回的文件ID
     * @throws IOException 文件处理异常
     * @throws OotpException 通常是 Http 请求异常，或者服务器返回错误消息
     */
    default String uploadToFsStorage(String repo, String contentType, InputStream file, String fileName, long fileSize, long lastModified)
            throws IOException, OotpException {
        return upload(STORAGE_FS, repo, contentType, file, fileName, fileSize, lastModified);
    }

    /**
     * 上传文件到文件服务文件库。
     *
     * @param repo 目标文件库
     * @param contentType 文件内容类型
     * @param file 文件内容
     * @param fileName 文件名
     * @param fileSize 文件大小，未知大小传 -1
     * @param lastModified 文件最后更新时间
     * @param pathInfo 文件上传的后置处理
     * @return 上传后返回的文件ID
     * @throws IOException 文件处理异常
     * @throws OotpException 通常是 Http 请求异常，或者服务器返回错误消息
     */
    default String uploadToFsStorage(String repo, String contentType, InputStream file, String fileName, long fileSize, long lastModified, String pathInfo)
            throws IOException, OotpException {
        return upload(STORAGE_FS, repo, contentType, file, fileName, fileSize, lastModified, pathInfo);
    }

    /**
     * 上传文件到默认文件中转服务。
     * @param to 目标接收方
     * @param contentType 文件内容类型
     * @param file 文件
     * @return 上传后返回的文件ID
     * @throws IOException 文件处理异常
     * @throws OotpException 其它异常
     */
    default String uploadToDefaultStorage(String to, String contentType, File file) throws IOException, OotpException {
        return upload(STORAGE_DEFAULT, to, contentType, file);
    }

    /**
     * 上传文件到默认文件中转服务。
     * @param to 目标接收方
     * @param contentType 文件内容类型
     * @param file 文件
     * @param pathInfo 文件上传的后置处理
     * @return 上传后返回的文件ID
     * @throws IOException 文件处理异常
     * @throws OotpException 其它异常
     */
    default String uploadToDefaultStorage(String to, String contentType, File file, String pathInfo) throws IOException, OotpException {
        return upload(STORAGE_DEFAULT, to, contentType, file, pathInfo);
    }

    /**
     * 上传文件到默认文件中转服务。
     *
     * @param to 目标接收方
     * @param contentType 文件内容类型
     * @param file 文件内容
     * @param fileName 文件名
     * @param fileSize 文件大小，未知大小传 -1
     * @param lastModified 文件最后更新时间
     * @return 上传后返回的文件ID
     * @throws IOException 文件处理异常
     * @throws OotpException 其它异常
     */
    default String uploadToDefaultStorage(String to, String contentType, InputStream file, String fileName, long fileSize, long lastModified)
            throws IOException, OotpException {
        return upload(STORAGE_DEFAULT, to, contentType, file, fileName, fileSize, lastModified);
    }

    /**
     * 上传文件到默认文件中转服务。
     *
     * @param to 目标接收方
     * @param contentType 文件内容类型
     * @param file 文件内容
     * @param fileName 文件名
     * @param fileSize 文件大小，未知大小传 -1
     * @param lastModified 文件最后更新时间
     * @param pathInfo 文件上传的后置处理
     * @return 上传后返回的文件ID
     * @throws IOException 文件处理异常
     * @throws OotpException 其它异常
     */
    default String uploadToDefaultStorage(String to, String contentType, InputStream file, String fileName, long fileSize, long lastModified, String pathInfo)
            throws IOException, OotpException {
        return upload(STORAGE_DEFAULT, to, contentType, file, fileName, fileSize, lastModified, pathInfo);
    }

    /**
     * 上传到个性化的存储器中。
     *
     * @param storage 存储器名称
     * @param contentType 文件内容类型
     * @param file 文件
     * @return 上传后返回的文件ID
     * @throws IOException 文件处理异常
     * @throws OotpException 其它异常
     */
    default String uploadToCustomStorage(String storage, String contentType, File file) throws IOException, OotpException {
        return upload(storage, null, contentType, file);
    }

    /**
     * 上传到个性化的存储器中。
     *
     * @param storage 存储器名称
     * @param contentType 文件内容类型
     * @param file 文件
     * @param pathInfo 文件上传的后置处理
     * @return 上传后返回的文件ID
     * @throws IOException 文件处理异常
     * @throws OotpException 其它异常
     */
    default String uploadToCustomStorage(String storage, String contentType, File file, String pathInfo) throws IOException, OotpException {
        return upload(storage, null, contentType, file, pathInfo);
    }

    /**
     * 上传到个性化的存储器中。
     * @param storage 存储器名称
     * @param contentType 文件内容类型
     * @param file 文件内容
     * @param fileName 文件名
     * @param fileSize 文件大小，未知大小传 -1
     * @param lastModified 文件最后更新时间
     * @return 上传后返回的文件ID
     * @throws IOException 文件处理异常
     * @throws OotpException 其它异常
     */
    default String uploadToCustomStorage(String storage, String contentType, InputStream file, String fileName, long fileSize, long lastModified)
            throws IOException, OotpException {
        return upload(storage, null, contentType, file, fileName, fileSize, lastModified);
    }

    /**
     * 上传到个性化的存储器中。
     * @param storage 存储器名称
     * @param contentType 文件内容类型
     * @param file 文件内容
     * @param fileName 文件名
     * @param fileSize 文件大小，未知大小传 -1
     * @param lastModified 文件最后更新时间
     * @param pathInfo 文件上传的后置处理
     * @return 上传后返回的文件ID
     * @throws IOException 文件处理异常
     * @throws OotpException 其它异常
     */
    default String uploadToCustomStorage(String storage, String contentType, InputStream file, String fileName, long fileSize, long lastModified, String pathInfo)
            throws IOException, OotpException {
        return upload(storage, null, contentType, file, fileName, fileSize, lastModified, pathInfo);
    }

    /**
     * 上传文件。
     * @param storage 存储器的名称
     * @param to 目标接收方
     * @param contentType 文件内容类型
     * @param file 文件
     * @return 上传后返回的文件ID
     * @throws IOException 文件处理异常
     * @throws OotpException 其它异常
     */
    default String upload(String storage, String to, String contentType, File file) throws IOException, OotpException {
        return upload(storage, to, contentType, file, null);
    }

    /**
     * 上传文件。
     * @param storage 存储器的名称
     * @param to 目标接收方
     * @param contentType 文件内容类型
     * @param file 文件
     * @param pathInfo 文件上传的后置处理
     * @return 上传后返回的文件ID
     * @throws IOException 文件处理异常
     * @throws OotpException 其它异常
     */
    default String upload(String storage, String to, String contentType, File file, String pathInfo) throws IOException, OotpException {
        try (FileInputStream fis = new FileInputStream(file)) {
            return upload(storage, to, contentType, fis, file.getName(), file.length(), file.lastModified(), pathInfo);
        }
    }

    /**
     * 上传文件。
     *
     * @param storage 存储器的名称
     * @param to 目标接收方，接收方ID或者文件库ID
     * @param contentType 文件内容类型
     * @param file 文件内容
     * @param fileName 文件名
     * @param fileSize 文件大小，未知大小传 -1
     * @param lastModified 文件最后更新时间
     * @return 上传后返回的文件ID
     * @throws IOException 文件处理异常
     * @throws OotpException 其它异常
     */
    default String upload(String storage, String to, String contentType, InputStream file, String fileName, long fileSize, long lastModified)
            throws IOException, OotpException {
        return upload(storage, to, contentType, file, fileName, fileSize, lastModified, null);
    }

    /**
     * 上传文件。
     *
     * @param storage 存储器的名称
     * @param to 目标接收方，接收方ID或者文件库ID
     * @param contentType 文件内容类型
     * @param file 文件内容
     * @param fileName 文件名
     * @param fileSize 文件大小，未知大小传 -1
     * @param lastModified 文件最后更新时间
     * @param pathInfo 文件后置处理路径
     * @return 上传后返回的文件ID
     * @throws IOException 文件处理异常
     * @throws OotpException 其它异常
     */
    String upload(String storage, String to, String contentType, InputStream file, String fileName, long fileSize, long lastModified, String pathInfo)
            throws IOException, OotpException;

    /**
     * 下载文件。仅上传到默认中转服务的文件可以下载。
     *
     * @param fileId 文件ID
     * @return 文件流
     */
    InputStream download(String fileId);

    /**
     * 复制到本地文件。仅上传到默认中转服务的文件可以复制。
     * @param fileId 文件ID
     * @param destination 本地文件
     * @throws IOException io 异常
     */
    default void copy(String fileId, File destination) throws IOException {
//        try (InputStream inputStream = download(fileId);
//             ReadableByteChannel rbc = Channels.newChannel(inputStream);
//             FileOutputStream fileOutputStream = new FileOutputStream(destination)) {
//            fileOutputStream.getChannel().transferFrom(rbc, 0, Long.MAX_VALUE);
//        }

        try (FileOutputStream fos = new FileOutputStream(destination)) {
            copy(fileId, fos);
        }
    }

    /**
     * 复制到本地文件。仅上传到默认中转服务的文件可以复制。
     * @param fileId 文件ID
     * @param destination 本地文件
     * @throws IOException io 异常
     */
    default void copy(String fileId, Path destination) throws IOException {
        try (InputStream inputStream = download(fileId)) {
            Files.copy(inputStream, destination, StandardCopyOption.REPLACE_EXISTING);
        }
    }

    /**
     * 复制到本地输出。仅上传到默认中转服务的文件可以复制。
     * @param fileId 文件ID
     * @param outputStream 输出流
     * @throws IOException io 异常
     */
    default void copy(String fileId, OutputStream outputStream) throws IOException {
        try (InputStream inputStream = download(fileId)) {
            StreamUtils.copy(inputStream, outputStream);
        }
    }

}
