package org.opoo.ootp.client.impl;

import lombok.extern.slf4j.Slf4j;
import org.opoo.ootp.client.OotpClient;
import org.opoo.ootp.client.OotpException;
import org.opoo.ootp.signer.Signer;
import org.springframework.core.io.InputStreamResource;
import org.springframework.http.HttpMethod;
import org.springframework.http.MediaType;
import org.springframework.http.RequestEntity;
import org.springframework.http.ResponseEntity;
import org.springframework.util.StreamUtils;
import org.springframework.web.client.ResponseExtractor;
import org.springframework.web.client.RestTemplate;

import java.io.File;
import java.io.FilterInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URI;
import java.net.URLEncoder;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.time.Instant;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

@Slf4j
public abstract class AbstractOotpClient implements OotpClient {
    public static final String RFC822_FORMAT = "EEE, dd MMM yyyy HH:mm:ss 'GMT'";
    public static DateTimeFormatter RFC822 = DateTimeFormatter.ofPattern(RFC822_FORMAT).withZone(ZoneId.of("GMT")).withLocale(Locale.US);
    public static final String FILE_ID_FIELD = "fileId";

    protected final RestTemplate restTemplate;

    AbstractOotpClient(RestTemplate restTemplate) {
        this.restTemplate = restTemplate;
    }

    public RestTemplate getRestTemplate() {
        return restTemplate;
    }

    @SuppressWarnings("rawtypes")
    @Override
    public String upload(String storage, String to, String contentType, InputStream file, String fileName,
                         long fileSize, long lastModified, String pathInfo) throws IOException, OotpException {
        Objects.requireNonNull(storage, "storage is required.");
        Objects.requireNonNull(contentType, "contentType is required.");
        Objects.requireNonNull(file, "file is required.");
        Objects.requireNonNull(fileName, "fileName is required.");

        String path = "/upload";
        if (pathInfo != null) {
            path += pathInfo;
        }
        path += "?storage=" + storage;

        final RequestEntity.BodyBuilder builder = RequestEntity.put(URI.create(path))
                .contentType(MediaType.parseMediaType(contentType))
                //.header(Signer.HEADER_NAME_PREFIX + "repo", repo)
                .header(Signer.HEADER_NAME_PREFIX + "meta-file-name", URLEncoder.encode(fileName, "UTF-8"));

        if (STORAGE_FS.equals(storage)) {
            String repo = Objects.requireNonNull(to, "repo is required.");
            builder.header(Signer.HEADER_NAME_PREFIX + "repo", repo);
        } else if (STORAGE_DEFAULT.equals(storage)) {
            Objects.requireNonNull(to, "to is required.");
            builder.header(Signer.HEADER_NAME_PREFIX + "exs-to", to);
        }

        if (lastModified > 0) {
            builder.header(Signer.HEADER_NAME_PREFIX + "meta-last-modified", RFC822.format(Instant.ofEpochMilli(lastModified)));
        }

        if (fileSize > 0) {
            builder.contentLength(fileSize);
        }

        final RequestEntity<InputStreamResource> requestEntity = builder.body(new InputStreamResource(file));
        final ResponseEntity<Map> responseEntity = restTemplate.exchange(requestEntity, Map.class);

        return Optional.ofNullable(responseEntity.getBody()).map(m -> (String) m.get(FILE_ID_FIELD))
                .orElseThrow(() -> new OotpException("响应内容为空", 500, "ResponseEmptyBody"));
    }

    protected <T> T downloadInternal(String fileId, ResponseExtractor<T> extractor) {
        Objects.requireNonNull(fileId, "fileId is required.");
        return restTemplate.execute("/file/" + fileId, HttpMethod.GET, null, extractor);
    }

    @Override
    public InputStream download(String fileId) {
        return downloadInternal(fileId, response -> OotpFileCachedInputStream.of(response.getBody(), fileId));
    }

    @Override
    public void copy(String fileId, File destination) throws IOException {
        downloadInternal(fileId, response -> {
            Files.copy(response.getBody(), destination.toPath(), StandardCopyOption.REPLACE_EXISTING);
            return null;
        });
    }

    @Override
    public void copy(String fileId, Path destination) throws IOException {
        downloadInternal(fileId, response -> {
            Files.copy(response.getBody(), destination, StandardCopyOption.REPLACE_EXISTING);
            return null;
        });
    }

    @Override
    public void copy(String fileId, OutputStream outputStream) throws IOException {
        downloadInternal(fileId, response -> {
            StreamUtils.copy(response.getBody(), outputStream);
            return null;
        });
    }

    @Slf4j
    static class OotpFileCachedInputStream extends FilterInputStream {
        private final Path file;
        /**
         * Creates a <code>FilterInputStream</code>
         * by assigning the  argument <code>in</code>
         * to the field <code>this.in</code> so as
         * to remember it for later use.
         *
         * @param in the underlying input stream, or <code>null</code> if
         *           this instance is to be created without an underlying stream.
         * @param file cached file.
         */
        protected OotpFileCachedInputStream(InputStream in, Path file) {
            super(in);
            this.file = file;
        }

        @Override
        public void close() throws IOException {
            super.close();
            final boolean result = Files.deleteIfExists(file);
            log.debug("关闭时删除临时文件：{} - {}", file, result);
        }

        static OotpFileCachedInputStream of(InputStream originalInputStream, String fileId) throws IOException {
            final Path path = Files.createTempFile("ootp", fileId);
            log.debug("复制到临时文件：{}", path);
            Files.copy(originalInputStream, path, StandardCopyOption.REPLACE_EXISTING);
            return new OotpFileCachedInputStream(Files.newInputStream(path), path);
        }
    }
}
