package org.opoo.ootp.client.impl;


import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.Data;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.opoo.ootp.client.ExsClient;
import org.opoo.ootp.client.ExsMessage;
import org.opoo.ootp.client.ExsMessageContent;
import org.opoo.ootp.client.OotpException;
import org.opoo.ootp.signer.Signer;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.RequestEntity;
import org.springframework.http.ResponseEntity;
import org.springframework.http.converter.json.Jackson2ObjectMapperBuilder;
import org.springframework.web.client.RestTemplate;

import java.net.URI;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

@Slf4j
class ExsClientV1 extends AbstractOotpClient implements ExsClient {
    private final ObjectMapper objectMapper;

    ExsClientV1(RestTemplate restTemplate) {
        super(restTemplate);
        this.objectMapper = Jackson2ObjectMapperBuilder.json()
                .failOnEmptyBeans(false)
                .failOnUnknownProperties(false)
                .build();
    }

    @Override
    public ExsClient getExsClient() {
        return this;
    }

    @Override
    public void send(String type, String to, String contentType, String body) throws OotpException {
        sendInternal(type, to, contentType, body);
    }

    @Override
    public void send(String type, String to, String contentType, byte[] body) throws OotpException {
        sendInternal(type, to, contentType, body);
    }
    
    void sendInternal(String type, String to, String contentType, Object body) throws OotpException {
        Objects.requireNonNull(type, "type is required.");
        Objects.requireNonNull(to, "to is required.");
        Objects.requireNonNull(contentType, "contentType is required.");
        Objects.requireNonNull(body, "body is required.");

        final RequestEntity<Object> requestEntity = RequestEntity.put(URI.create("/exs/api/messages/v1"))
                .header(Signer.HEADER_NAME_PREFIX + "exs-type", type)
                .header(Signer.HEADER_NAME_PREFIX + "exs-to", to)
                .contentType(MediaType.parseMediaType(contentType))
                .body(body);
        final ResponseEntity<MessageWrapper> responseEntity = restTemplate.exchange(requestEntity, MessageWrapper.class);

        final MessageWrapper messageWrapper = Optional.ofNullable(responseEntity.getBody())
                .orElseThrow(() -> new OotpException("响应内容为空", 500, "ResponseEmptyBody"));

        log.debug("send result: {}", messageWrapper);

        if (!messageWrapper.isSuccess()) {
            throw new OotpException(messageWrapper.getErrMsg(), messageWrapper.getCode(), messageWrapper.getErrMsg());
        }
    }

    @Override
    public List<ExsMessage> poll(int limit, boolean autoAck, boolean full) throws OotpException {
        if (limit <= 0 || limit > DEFAULT_LIMIT) {
            limit = DEFAULT_LIMIT;
        }

        Map<String, Object> request = new LinkedHashMap<>();
        request.put("limit", limit);
        request.put("autoAck", autoAck);
        request.put("full", full);

        final MessageWrapper messageWrapper = restTemplate.postForObject("/exs/api/messages/v1", request, MessageWrapper.class);

        if (messageWrapper == null) {
            throw new OotpException("响应内容为空", 500, "ResponseEmptyBody");
        }

        if (!messageWrapper.isSuccess()) {
            throw new OotpException(messageWrapper.getErrMsg(), messageWrapper.getCode(), messageWrapper.getErrMsg());
        }

        return Optional.ofNullable(messageWrapper.getResultData())
                .orElseGet(ArrayList::new)
                .stream()
                .map(ExsClientV1::toMessage).collect(Collectors.toList());
    }

    @SneakyThrows
    @Override
    public ExsMessageContent getMessageContent(String id) throws OotpException {
        Objects.requireNonNull(id, "id is required.");

        final ResponseEntity<String> responseEntity = restTemplate.getForEntity("/exs/api/messages/" + id + "/v1", String.class);
        final HttpHeaders headers = responseEntity.getHeaders();
        final String body = Optional.ofNullable(responseEntity.getBody())
                .orElseThrow(() -> new OotpException("响应内容为空", 500, "ResponseEmptyBody"));
        final MediaType contentType = Optional.ofNullable(headers.getContentType())
                .orElseThrow(() -> new OotpException("响应缺少类型", 500, "ResponseNoContentType"));

        // json
        if (MediaType.APPLICATION_JSON.isCompatibleWith(contentType)) {
            final Map<?, ?> map = objectMapper.readValue(body, Map.class);
            Object success = map.get("success");
            if (success instanceof Boolean && !(boolean) success) {
                throw new OotpException((String) map.get("errMsg"));
            }
        }

        return new ExsMessageContent(contentType.toString(), body);
    }

    @Override
    public void ack(List<String> ids) throws OotpException {
        if (ids == null || ids.isEmpty()) {
            throw new IllegalArgumentException("ids 不能为空");
        }

        Map<String, Object> request = new LinkedHashMap<>();
        if (ids.size() == 1) {
            request.put("id", ids.iterator().next());
        } else {
            request.put("ids", ids);
        }

        final MessageWrapper messageWrapper = restTemplate.postForObject("/exs/api/messages/ack/v1", request, MessageWrapper.class);

        if (messageWrapper == null) {
            throw new OotpException("响应内容为空", 500, "ResponseEmptyBody");
        }

        log.debug("ack result: {}", messageWrapper);

        if (!messageWrapper.isSuccess()) {
            throw new OotpException(messageWrapper.getErrMsg(), messageWrapper.getCode(), messageWrapper.getErrMsg());
        }
    }

    private static ExsMessage toMessage(MessageV1 messageV1) {
        final ExsMessage exsMessage = new ExsMessage();
        exsMessage.setId(messageV1.getId());
        exsMessage.setType(messageV1.getBizType());
        exsMessage.setFrom(messageV1.getFromSysId());
        exsMessage.setContentType(messageV1.getContentType());
        exsMessage.setBody(messageV1.getMsgBody());
        return exsMessage;
    }

    /**
     * 简直了，不知道谁定义的数据结构，多了一层。
     */
    @Data
    public static class MessageWrapper {
        private boolean success;
        private int code;
        private String errMsg;
        private List<MessageV1> resultData;
    }

    /**
     * V1 版接口的消息
     */
    @Data
    public static class MessageV1 {
        private String id;
        private String bizType;
        private String fromSysId;
        private String contentType;

        private String msgBody;
    }
}
