package org.opoo.ootp.client.impl;


import lombok.extern.slf4j.Slf4j;
import org.opoo.ootp.client.ExsClient;
import org.opoo.ootp.client.ExsMessage;
import org.opoo.ootp.client.ExsMessageContent;
import org.opoo.ootp.client.OotpClient;
import org.opoo.ootp.client.OotpException;
import org.opoo.ootp.signer.Signer;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.RequestEntity;
import org.springframework.http.ResponseEntity;
import org.springframework.web.client.RestTemplate;

import java.net.URI;
import java.util.Arrays;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

@Slf4j
class ExsClientV2 extends AbstractOotpClient implements ExsClient, OotpClient {
    
    ExsClientV2(RestTemplate restTemplate) {
        super(restTemplate);
    }

    @Override
    public ExsClient getExsClient() {
        return this;
    }

    @Override
    public void send(String type, String to, String contentType, String body) throws OotpException {
        sendInternal(type, to, contentType, body);
    }

    @Override
    public void send(String type, String to, String contentType, byte[] body) throws OotpException {
        sendInternal(type, to, contentType, body);
    }

    void sendInternal(String type, String to, String contentType, Object body) throws OotpException {
        Objects.requireNonNull(type, "type is required.");
        Objects.requireNonNull(to, "to is required.");
        Objects.requireNonNull(contentType, "contentType is required.");
        Objects.requireNonNull(body, "body is required.");

        final RequestEntity<Object> requestEntity = RequestEntity.put(URI.create("/exs/api/messages/v2"))
                .header(Signer.HEADER_NAME_PREFIX + "exs-type", type)
                .header(Signer.HEADER_NAME_PREFIX + "exs-to", to)
                .contentType(MediaType.parseMediaType(contentType))
                .body(body);

        final ResponseEntity<String> responseEntity = restTemplate.exchange(requestEntity, String.class);
        log.debug("send result: {}", responseEntity.getBody());
    }

    @Override
    public List<ExsMessage> poll(int limit, boolean autoAck, boolean full) throws OotpException {
        if (limit <= 0 || limit > DEFAULT_LIMIT) {
            limit = DEFAULT_LIMIT;
        }

        Map<String, Object> request = new LinkedHashMap<>();
        request.put("limit", limit);
        request.put("autoAck", autoAck);
        request.put("full", full);

        final ExsMessage[] messages = restTemplate.postForObject("/exs/api/messages/v2", request, ExsMessage[].class);
        return Arrays.asList(Objects.requireNonNull(messages));
    }

    @Override
    public ExsMessageContent getMessageContent(String id) throws OotpException {
        Objects.requireNonNull(id, "id is required.");

        final ResponseEntity<String> responseEntity = restTemplate.getForEntity("/exs/api/messages/" + id + "/v2", String.class);
        final HttpHeaders headers = responseEntity.getHeaders();
        final String body = Optional.ofNullable(responseEntity.getBody())
                .orElseThrow(() -> new OotpException("响应内容为空", 500, "ResponseEmptyBody"));
        final MediaType contentType = Optional.ofNullable(headers.getContentType())
                .orElseThrow(() -> new OotpException("响应缺少类型", 500, "ResponseNoContentType"));

        return new ExsMessageContent(contentType.toString(), body);
    }

    @Override
    public void ack(List<String> ids) throws OotpException {
        if (ids == null || ids.isEmpty()) {
            throw new IllegalArgumentException("ids 不能为空");
        }

        Map<String, Object> request = new LinkedHashMap<>();
        if (ids.size() == 1) {
            request.put("id", ids.iterator().next());
        } else {
            request.put("ids", ids);
        }

        final String result = restTemplate.postForObject("/exs/api/messages/ack/v2", request, String.class);
        log.debug("ack result: {}", result);
    }

}
