package org.opoo.ootp.client.impl;

import lombok.extern.slf4j.Slf4j;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.opoo.ootp.client.OotpClient;
import org.opoo.ootp.signer.SM3Signer;
import org.opoo.ootp.signer.Signer;
import org.opoo.ootp.signer.spring.SignerClientHttpRequestInterceptor;
import org.springframework.http.HttpRequest;
import org.springframework.http.client.ClientHttpRequestFactory;
import org.springframework.http.client.ClientHttpRequestInterceptor;
import org.springframework.http.client.HttpComponentsClientHttpRequestFactory;
import org.springframework.http.client.support.HttpRequestWrapper;
import org.springframework.util.Assert;
import org.springframework.util.ClassUtils;
import org.springframework.web.client.RestTemplate;
import org.springframework.web.util.UriComponentsBuilder;

import java.net.URI;
import java.util.List;

@Slf4j
public class OotpClientBuilder {
    private URI endpoint;
    private Signer signer;
    private RestTemplate restTemplate;
    private boolean version2 = false;
    private String accessKey;
    private String secretKey;

    private OotpClientBuilder() {
    }

    public static OotpClient defaultClient(URI endpoint, Signer signer) {
        return new OotpClientBuilder().endpoint(endpoint).signer(signer).build();
    }

    public static OotpClient defaultClient(String endpoint, Signer signer) {
        return defaultClient(URI.create(endpoint), signer);
    }

    public static OotpClientBuilder custom() {
        return new OotpClientBuilder();
    }

    public OotpClientBuilder credentials(String accessKey, String secretKey) {
        this.accessKey = accessKey;
        this.secretKey = secretKey;
        return this;
    }

    public OotpClientBuilder signer(Signer signer) {
        this.signer = signer;
        return this;
    }

    public OotpClientBuilder endpoint(URI endpoint) {
        this.endpoint = endpoint;
        return this;
    }

    public OotpClientBuilder endpoint(String endpoint) {
        this.endpoint = URI.create(endpoint);
        return this;
    }

    /**
     * RestTemplate 必须是专用的，会增加拦截器。
     *
     * @param restTemplate Spring RestTemplate
     * @return 当前 Builder
     */
    public OotpClientBuilder restTemplate(RestTemplate restTemplate) {
        this.restTemplate = restTemplate;
        return this;
    }

    public OotpClientBuilder useVersion2() {
        this.version2 = true;
        return this;
    }

    public RestTemplate buildRestTemplate() {
        Assert.notNull(endpoint, "必须设置 endpoint");

        Signer signer = this.signer;
        RestTemplate restTemplate = this.restTemplate;

        if (signer == null) {
            if (accessKey == null || secretKey == null) {
                throw new IllegalArgumentException("必须设置 signer 或者 accessKey/secretKey");
            }
            log.debug("默认使用 SMSigner");
            signer = new SM3Signer(accessKey, secretKey);
        }

        if (restTemplate == null) {
            restTemplate = new RestTemplate();

            if (ClassUtils.isPresent("org.apache.http.impl.client.HttpClients", OotpClientBuilder.class.getClassLoader())) {
                log.debug("Using Apache HttpComponents as ClientRequestFactory ...");
                restTemplate.setRequestFactory(HttpComponentsClientHttpRequestFactoryCreator.create());
            }
        }

        final List<ClientHttpRequestInterceptor> interceptors = restTemplate.getInterceptors();
        interceptors.add((request, body, execution) -> execution.execute(new EndpointProcessingHttpRequestWrapper(request, endpoint), body));
        interceptors.add(new SignerClientHttpRequestInterceptor(signer));
        return restTemplate;
    }

    public OotpClient build() {
        RestTemplate restTemplate = buildRestTemplate();
        return version2 ? new ExsClientV2(restTemplate) : new ExsClientV1(restTemplate);
    }

    static class HttpComponentsClientHttpRequestFactoryCreator {
        static ClientHttpRequestFactory create() {
            final CloseableHttpClient httpClient = HttpClients.custom()
                    .disableAuthCaching()
                    .disableCookieManagement()
                    .disableRedirectHandling()
                    .setMaxConnPerRoute(20)
                    .setMaxConnTotal(20)
                    .build();
            return new HttpComponentsClientHttpRequestFactory(httpClient);
        }
    }

    static class EndpointProcessingHttpRequestWrapper extends HttpRequestWrapper {
        private final URI endpoint;
        private URI uri;

        /**
         * Create a new {@code HttpRequest} wrapping the given request object.
         *
         * @param request the request object to be wrapped
         */
        public EndpointProcessingHttpRequestWrapper(HttpRequest request, URI endpoint) {
            super(request);
            this.endpoint = endpoint;
        }

        @Override
        public URI getURI() {
            if (uri == null) {
                uri = processUri(super.getURI());
            }

            return uri;
        }

        private URI processUri(URI originalUri) {
            if (originalUri.isAbsolute()) {
                return originalUri;
            }

            final URI uri = UriComponentsBuilder.fromUri(originalUri)
                    .scheme(endpoint.getScheme())
                    .host(endpoint.getHost())
                    .port(endpoint.getPort())
                    .build().toUri();

            log.debug("URI 补全：{} -> {}", originalUri, uri);
            return uri;
        }
    }
}
