package org.opoo.ootp.client.impl;

import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.extern.slf4j.Slf4j;
import org.apache.http.HttpEntity;
import org.apache.http.HttpHost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.client.utils.URIUtils;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.InputStreamEntity;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.opoo.ootp.client.EntityBody;
import org.opoo.ootp.client.ExsBody;
import org.opoo.ootp.client.ExsCodec;
import org.opoo.ootp.client.ExsMetadata;
import org.opoo.ootp.client.MessageClient;
import org.opoo.ootp.client.OotpException;
import org.opoo.ootp.client.StringBody;

import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.Optional;

@Slf4j
public abstract class AbstractMessageClient implements MessageClient {
    private static final String DEFAULT_BASE_PATH = "/exs-api";

    private final URI endpoint ;
    protected final CloseableHttpClient httpClient;
    protected final ObjectMapper objectMapper;
    protected final String basePath;
    private ExsCodec codec;

    public AbstractMessageClient(URI endpoint, CloseableHttpClient httpClient, ObjectMapper objectMapper, String basePath) {
        this.endpoint = endpoint;
        this.httpClient = httpClient;
        this.objectMapper = objectMapper;
        this.basePath = basePath != null ? basePath : DEFAULT_BASE_PATH;
    }

    public ExsCodec getCodec() {
        return codec;
    }

    public void setCodec(ExsCodec codec) {
        this.codec = codec;
    }

    protected void processUri(HttpUriRequest request) {
        try {
            final URI uri = request.getURI();
            final URI newUri = URIUtils.rewriteURI(uri, new HttpHost(endpoint.getHost(), endpoint.getPort(), endpoint.getScheme()));
            ((HttpRequestBase) request).setURI(newUri);
            log.debug("URI 补全到：{} -> {}", uri, newUri);
        } catch (URISyntaxException ex) {
            throw new OotpException("URI 处理失败", ex);
        }
    }

    protected static HttpEntity toEntity(ExsBody body, ExsMetadata metadata) throws IOException {
        if (body instanceof EntityBody) {
            return ((EntityBody) body).getEntity();
        }

        final ContentType contentType = ContentType.parse(metadata.getContentType());
        if (body instanceof StringBody) {
            final StringBody stringBody = (StringBody) body;
            ContentType contentType2 = Optional.ofNullable(stringBody.getCharset()).map(contentType::withCharset).orElse(contentType);
            return new StringEntity(stringBody.getString(), contentType2);
        }

        final long contentLength = Optional.ofNullable(metadata.getContentLength()).filter(l -> l > 0).orElse(-1L);
        return new InputStreamEntity(body.getContent(), contentLength, contentType);
    }

    protected HttpEntity codecEncode(HttpEntity entity, ExsMetadata metadata) throws IOException {
        if (codec == null) {
            log.debug("ExsCodec 没有配置，消息无需（无法）编码/加密");
            return entity;
        }

        return codec.encode(entity, metadata);
    }

    protected HttpEntity codecDecode(HttpEntity entity, ExsMetadata metadata) throws IOException {
        if (codec == null) {
            log.debug("ExsCodec 没有配置，消息无需（无法）解码/解密");
            return entity;
        }

        return codec.decode(entity, metadata);
    }
}
