package org.opoo.ootp.client.impl;

import com.fasterxml.jackson.databind.ObjectMapper;
import org.apache.http.HttpEntity;
import org.apache.http.HttpStatus;
import org.apache.http.StatusLine;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPut;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.InputStreamEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.util.EntityUtils;
import org.opoo.ootp.client.EntityBody;
import org.opoo.ootp.client.ExsBody;
import org.opoo.ootp.client.ExsCodec;
import org.opoo.ootp.client.ExsMessage;
import org.opoo.ootp.client.ExsMetadata;
import org.opoo.ootp.client.FileClient;
import org.opoo.ootp.client.Metadata;
import org.opoo.ootp.client.OotpException;

import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.nio.file.Files;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

public class FileClientImpl implements FileClient {
    public static final String FILE_ID_FIELD = "fileId";

    private final URI endpoint;
    private final CloseableHttpClient httpClient;
    private final ObjectMapper objectMapper;
    private ExsCodec codec;

    public FileClientImpl(URI endpoint, CloseableHttpClient httpClient, ObjectMapper objectMapper) {
        this.endpoint = endpoint;
        this.httpClient = httpClient;
        this.objectMapper = objectMapper;
    }

    public ExsCodec getCodec() {
        return codec;
    }

    public void setCodec(ExsCodec codec) {
        this.codec = codec;
    }

    @Override
    public String upload(ExsMessage message, String storage, String pathInfo) throws IOException, OotpException {
        final ExsBody body = message.getBody();
        final ExsMetadata metadata = message.getMetadata();
        Objects.requireNonNull(storage, "storage is required.");
        Objects.requireNonNull(metadata, "metadata is required.");
        Objects.requireNonNull(metadata.getContentType(), "contentType is required.");
        Objects.requireNonNull(body, "file is required.");
        Objects.requireNonNull(metadata.getUserMetadata(Metadata.META_FILE_NAME), "fileName is required.");

        if (STORAGE_FS.equals(storage)) {
            Objects.requireNonNull(metadata.getTo(), "repo is required.");
        } else if (STORAGE_DEFAULT.equals(storage)) {
            Objects.requireNonNull(metadata.getTo(), "to is required.");
        }

        String path = endpoint.toString() + "/file-api/upload";
        if (pathInfo != null) {
            path += pathInfo;
        }
        path += "?storage=" + storage;

        final HttpEntity entity2 = AbstractMessageClient.toEntity(body, metadata);
        final HttpEntity entity = codec != null ? codec.encode(entity2, metadata) : entity2;

        final HttpPut httpPut = new HttpPut(path);
        httpPut.setEntity(entity);
        metadata.toHeaders(httpPut::setHeader);

        try (final CloseableHttpResponse response = httpClient.execute(httpPut)) {
            final StatusLine statusLine = response.getStatusLine();
            final int statusCode = statusLine.getStatusCode();

            final Map<?,?> map;
            try (final InputStream inputStream = response.getEntity().getContent()) {
                map = objectMapper.readValue(inputStream, Map.class);
            }

            if (statusCode < HttpStatus.SC_OK || statusCode >= HttpStatus.SC_MULTIPLE_CHOICES) {
//                if (map != null && (map.containsKey("error") || map.containsKey("message"))) {
//                    final String error = (String) map.get("error");
//                    final String message2 = (String) map.get("message");
//                    throw new OotpException(message2 != null ? message2 : statusCode + "-" + error, statusCode, error);
//                }
                EntityUtils.consumeQuietly(response.getEntity());
                throw new OotpException("文件上传失败：" + statusCode, statusCode, statusLine.getReasonPhrase());
            }

            return Optional.ofNullable(map).map(m -> (String) m.get(FILE_ID_FIELD))
                    .orElseThrow(() -> new OotpException("响应内容为空", 500, "ResponseEmptyBody"));
        }
    }

    @Override
    public ExsMessage getFile(String fileId) throws IOException {
        Objects.requireNonNull(fileId, "fileId is required.");
        final HttpGet httpGet = new HttpGet(endpoint.toString() + "/file-api/file/" + fileId);
        try (final CloseableHttpResponse response = httpClient.execute(httpGet)) {
            final StatusLine statusLine = response.getStatusLine();
            final int statusCode = statusLine.getStatusCode();

            if (statusCode < HttpStatus.SC_OK || statusCode >= HttpStatus.SC_MULTIPLE_CHOICES) {
//                final Map<?,?> map;
//                try (final InputStream inputStream = response.getEntity().getContent()) {
//                    map = objectMapper.readValue(inputStream, Map.class);
//                }
//                if (map != null && (map.containsKey("error") || map.containsKey("message"))) {
//                    final String error = (String) map.get("error");
//                    final String message2 = (String) map.get("message");
//                    throw new OotpException(message2 != null ? message2 : statusCode + "-" + error, statusCode, error);
//                }
                EntityUtils.consumeQuietly(response.getEntity());
                throw new OotpException("文件下载失败：" + statusCode, statusCode, statusLine.getReasonPhrase());
            }

            final HttpEntity entity = response.getEntity();
            final ContentType contentType = ContentType.get(entity);

            final Map<String,String> headers = new HashMap<>();
            Arrays.stream(response.getAllHeaders()).forEach(h -> headers.putIfAbsent(h.getName().toLowerCase(), h.getValue()));
            final ExsMetadata metadata = ExsMetadata.fromHeaders(headers);
            metadata.setId(fileId);

            final InputStream content = entity.getContent();
            final InputStream decodeStream = codec != null ? codec.decode(content, metadata) : content;
            final FileCachedInputStream ootpFileCachedInputStream = FileCachedInputStream.create(decodeStream, fileId);
            final long size = Files.size(ootpFileCachedInputStream.getFile());
            final InputStreamEntity inputStreamEntity = new InputStreamEntity(ootpFileCachedInputStream, size, contentType);
            final EntityBody entityBody = new EntityBody(inputStreamEntity);

            return new ExsMessage(entityBody, metadata);
        }
    }

}
