package org.opoo.ootp.client.impl;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.extern.slf4j.Slf4j;
import org.apache.http.HttpEntity;
import org.apache.http.HttpHost;
import org.apache.http.HttpStatus;
import org.apache.http.NameValuePair;
import org.apache.http.StatusLine;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.client.utils.URIUtils;
import org.apache.http.entity.BufferedHttpEntity;
import org.apache.http.entity.ByteArrayEntity;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.InputStreamEntity;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.util.EntityUtils;
import org.opoo.ootp.client.EntityBody;
import org.opoo.ootp.client.ExsBody;
import org.opoo.ootp.client.ExsCodec;
import org.opoo.ootp.client.ExsMessage;
import org.opoo.ootp.client.ExsMetadata;
import org.opoo.ootp.client.MessageClient;
import org.opoo.ootp.client.OotpException;
import org.opoo.ootp.client.StringBody;
import org.opoo.ootp.codec.Codec;

import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Function;
import java.util.function.Supplier;

import static org.opoo.ootp.client.Metadata.HEADER_CONTENT_LENGTH;

@Slf4j
public abstract class AbstractMessageClient implements MessageClient {
    private static final String DEFAULT_BASE_PATH = "/exs-api";

    private final URI endpoint ;
    protected final CloseableHttpClient httpClient;
    protected final ObjectMapper objectMapper;
    protected final String basePath;
    private ExsCodec codec;

    public AbstractMessageClient(URI endpoint, CloseableHttpClient httpClient, ObjectMapper objectMapper, String basePath) {
        this.endpoint = endpoint;
        this.httpClient = httpClient;
        this.objectMapper = objectMapper;
        this.basePath = basePath != null ? basePath : DEFAULT_BASE_PATH;
    }

    public ExsCodec getCodec() {
        return codec;
    }

    public void setCodec(ExsCodec codec) {
        this.codec = codec;
    }

    protected void processUri(HttpUriRequest request) {
        try {
            final URI uri = request.getURI();
            final URI newUri = URIUtils.rewriteURI(uri, new HttpHost(endpoint.getHost(), endpoint.getPort(), endpoint.getScheme()));
            ((HttpRequestBase) request).setURI(newUri);
            log.debug("URI 补全到：{} -> {}", uri, newUri);
        } catch (URISyntaxException ex) {
            throw new OotpException("URI 处理失败", ex);
        }
    }

    protected static HttpEntity toEntity(ExsBody body, ExsMetadata metadata) throws IOException {
        if (body instanceof EntityBody) {
            return ((EntityBody) body).getEntity();
        }

        final ContentType contentType = ContentType.parse(metadata.getContentType());
        if (body instanceof StringBody) {
            final StringBody stringBody = (StringBody) body;
            ContentType contentType2 = Optional.ofNullable(stringBody.getCharset()).map(contentType::withCharset).orElse(contentType);
            return new StringEntity(stringBody.getString(), contentType2);
        }

        final long contentLength = Optional.ofNullable(metadata.getContentLength()).filter(l -> l > 0).orElse(-1L);
        return new InputStreamEntity(body.getContent(), contentLength, contentType);
    }

    protected HttpEntity codecEncode(HttpEntity entity, ExsMetadata metadata) throws IOException {
        if (codec == null) {
            log.debug("ExsCodec 没有配置，消息无需（无法）编码/加密");
            return entity;
        }

        return codec.encode(entity, metadata);
    }

    protected HttpEntity codecDecode(HttpEntity entity, ExsMetadata metadata) throws IOException {
        if (codec == null) {
            log.debug("ExsCodec 没有配置，消息无需（无法）解码/解密");
            return entity;
        }

        return codec.decode(entity, metadata);
    }

    protected void validateResponse(CloseableHttpResponse response) throws IOException, OotpException {
        validateResponse(response, () -> true, null);
    }

    /**
     *
     * @param response
     * @param parseErrorBody 是否需要解析错误消息的主体
     * @param baseExceptionMessage 错误消息基本消息
     * @throws IOException
     * @throws OotpException
     */
    protected void validateResponse(CloseableHttpResponse response, Supplier<Boolean> parseErrorBody, String baseExceptionMessage) throws IOException, OotpException {
        final StatusLine statusLine = response.getStatusLine();
        final HttpEntity entity = response.getEntity();
        final int statusCode = statusLine.getStatusCode();

        if (statusCode < HttpStatus.SC_OK || statusCode >= HttpStatus.SC_MULTIPLE_CHOICES) {
            if (hasContent(response) && parseErrorBody.get()) {
                // 一定会 close，无需再调用 EntityUtils.consume(HttpEntity)
                try (final InputStream inputStream = entity.getContent()) {
                    final Map<?,?> map = objectMapper.readValue(inputStream, Map.class);
                    final String error = (String) map.get("error");
                    final String message = (String) map.get("message");
                    final String errMsg = (String) map.get("errMsg");

                    if (errMsg != null) {
                        throw new OotpException(statusCode + " " + errMsg, statusCode, statusLine.getReasonPhrase());
                    }

                    String exceptionMessage = "消息API调用失败：" + statusCode;
                    if (error != null) {
                        exceptionMessage += " " + error;
                    }

                    if (message != null) {
                        exceptionMessage += " - " + message;
                    }

                    throw new OotpException(exceptionMessage, statusCode, error);
                }
            } else {
                log.debug("Response status: {}, content length: {}", statusCode, response.getFirstHeader(HEADER_CONTENT_LENGTH));
                EntityUtils.consumeQuietly(entity);
            }

            final String msg = Optional.ofNullable(baseExceptionMessage).orElse("消息API调用失败");
            throw new OotpException(msg + "：" + statusCode + " " + statusLine.getReasonPhrase(), statusCode, statusLine.getReasonPhrase());
        }
    }

    protected boolean hasContent(CloseableHttpResponse response) {
        final Long length = Optional.ofNullable(response.getFirstHeader(HEADER_CONTENT_LENGTH)).map(NameValuePair::getValue).map(Long::parseLong).orElse(0L);
        if (length > 0 ) {
            return true;
        }

        final String transferEncoding = Optional.ofNullable(response.getFirstHeader("Transfer-Encoding")).map(NameValuePair::getValue).orElse(null);
        return "chunked".equals(transferEncoding);
    }

    protected ExsMessage getMessage(String id, Function<String,String> idToPath) throws OotpException {
        Objects.requireNonNull(id, "id is required.");
        Objects.requireNonNull(idToPath, "idToPath function is required.");
        final HttpGet httpGet = new HttpGet(basePath + idToPath.apply(id));
        processUri(httpGet);
        try (final CloseableHttpResponse response = httpClient.execute(httpGet)) {
            final HttpEntity entity = response.getEntity();

            final Map<String,String> headers = new HashMap<>();
            Arrays.stream(response.getAllHeaders()).forEach(h -> headers.putIfAbsent(h.getName().toLowerCase(), h.getValue()));
            final ExsMetadata metadata = ExsMetadata.fromHeaders(headers);

            validateResponse(response, () -> {
                final ContentType contentType = ContentType.get(entity);
                if (contentType == null) {
                    return false;
                }
                final String str = contentType.toString().toLowerCase();
                // json 报文，且没有编码过，则可以解析
                return str.startsWith("application/") && str.contains("json") && metadata.getUserMetadata(Codec.META_TRANSFORM_MODE) == null;
            }, "获取消息[" + id + "]失败");

            final HttpEntity decodedEntity = codecDecode(entity, metadata);

            final BufferedHttpEntity httpEntity = new BufferedHttpEntity(decodedEntity);
            return new ExsMessage(new EntityBody(httpEntity), metadata);
        } catch (IOException ex) {
            throw new OotpException("获取消息[" + id + "]失败：" + ex.getMessage(), ex);
        }
    }

    protected int processIds(List<String> ids, String path, Function<HttpPost, Integer> resultHandler) throws OotpException {
        if (ids == null || ids.isEmpty()) {
            throw new IllegalArgumentException("ids 不能为空");
        }

        Map<String, Object> request = new LinkedHashMap<>();
        request.put("ids", ids);

        final byte[] bytes;
        try {
            bytes = objectMapper.writeValueAsBytes(request);
        } catch (JsonProcessingException e) {
            throw new OotpException("序列化 " + path + " 请求出错", e);
        }

        final HttpPost httpPost = new HttpPost(basePath + path);
        httpPost.setEntity(new ByteArrayEntity(bytes, ContentType.APPLICATION_JSON));

        return resultHandler.apply(httpPost);
    }
}
