package org.opoo.ootp.client.impl;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.util.VersionInfo;
import org.opoo.ootp.client.ExsCodec;
import org.opoo.ootp.client.FileClient;
import org.opoo.ootp.client.MessageClient;
import org.opoo.ootp.client.OotpClient;
import org.opoo.ootp.signer.Signer;
import org.opoo.ootp.signer.httpclient.SignerHttpRequestInterceptor;

import java.net.URI;
import java.util.Objects;
import java.util.Optional;

import static com.fasterxml.jackson.databind.DeserializationFeature.FAIL_ON_IGNORED_PROPERTIES;
import static com.fasterxml.jackson.databind.DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES;

@Slf4j
public class OotpClientBuilder {
    public static final String HTTP_VERSION = VersionInfo.getUserAgent("Apache-HttpClient", "org.apache.http.client", HttpClientBuilder.class);
    public static final String PACKAGE_VERSION = OotpClientBuilder.class.getPackage().getImplementationVersion();
    public static final String DEFAULT_USER_AGENT = String.format("Ootp Java Client%s %s (%s; %s; %s)",
            (PACKAGE_VERSION != null ? " v" + PACKAGE_VERSION : ""), HTTP_VERSION,
            System.getProperty("os.name"), System.getProperty("os.version"), System.getProperty("os.arch"));

    private URI endpoint;
    private Signer signer;
    private HttpClientBuilder httpClientBuilder;
    private CloseableHttpClient httpClient;
    private ObjectMapper objectMapper;
    private boolean version1 = false;
    private String basePath;
    private ExsCodec messageCodec;
    private ExsCodec fileCodec;

    private OotpClientBuilder() {
    }

    public static OotpClient defaultClient(URI endpoint, Signer signer) {
        return new OotpClientBuilder().endpoint(endpoint).signer(signer).build();
    }

    public static OotpClient defaultClient(String endpoint, Signer signer) {
        return defaultClient(URI.create(endpoint), signer);
    }

    public static OotpClientBuilder custom() {
        return new OotpClientBuilder();
    }

    public OotpClientBuilder signer(Signer signer) {
        this.signer = signer;
        return this;
    }

    public OotpClientBuilder endpoint(URI endpoint) {
        this.endpoint = endpoint;
        return this;
    }

    public OotpClientBuilder endpoint(String endpoint) {
        this.endpoint = URI.create(endpoint);
        return this;
    }

    public OotpClientBuilder basePath(String basePath) {
        this.basePath = basePath;
        return this;
    }

    public OotpClientBuilder objectMapper(ObjectMapper objectMapper) {
        this.objectMapper = objectMapper;
        return this;
    }

    /**
     * RestTemplate 必须是专用的，会增加拦截器。
     *
     * @param httpClientBuilder HttpClientBuilder
     * @return 当前 Builder
     */
    public OotpClientBuilder httpClientBuilder(HttpClientBuilder httpClientBuilder) {
        this.httpClientBuilder = httpClientBuilder;
        return this;
    }

    public OotpClientBuilder useVersion1() {
        this.version1 = true;
        return this;
    }

    public OotpClientBuilder messageCodec(ExsCodec codec) {
        this.messageCodec = codec;
        return this;
    }

    public OotpClientBuilder fileCodec(ExsCodec codec) {
        this.fileCodec = codec;
        return this;
    }

    public OotpClientBuilder httpClient(CloseableHttpClient httpClient) {
        this.httpClient = httpClient;
        return this;
    }

    public OotpClient build() {
        Objects.requireNonNull(endpoint, "必须设置 endpoint");

        final CloseableHttpClient httpClient = Optional.ofNullable(this.httpClient)
                .orElseGet(() -> buildDefaultOotpHttpClient(signer, this.httpClientBuilder));

        final ObjectMapper objectMapper = Optional.ofNullable(this.objectMapper)
                .orElseGet(() -> new ObjectMapper()
                        .disable(FAIL_ON_IGNORED_PROPERTIES)
                        .disable(FAIL_ON_UNKNOWN_PROPERTIES)
                        .disable(SerializationFeature.FAIL_ON_EMPTY_BEANS));

        final AbstractMessageClient messageClient = version1 ?
                new V1MessageClientImpl(endpoint, httpClient, objectMapper, basePath) :
                new V2MessageClientImpl(endpoint, httpClient, objectMapper, basePath);
        final FileClientImpl fileClient = new FileClientImpl(endpoint, httpClient, objectMapper);

        Optional.ofNullable(messageCodec).ifPresent(messageClient::setCodec);
        Optional.ofNullable(fileCodec).ifPresent(fileClient::setCodec);

        // 相同的  httpclient 则表明是外部注入的，不是自己生成不要自己关闭
        return new OotpClientImpl(messageClient, fileClient, (httpClient == this.httpClient) ? null : httpClient);
    }

    public static CloseableHttpClient buildDefaultOotpHttpClient(Signer signer) {
        return buildDefaultOotpHttpClient(signer, null);
    }

    public static CloseableHttpClient buildDefaultOotpHttpClient(Signer signer, HttpClientBuilder httpClientBuilder) {
        Objects.requireNonNull(signer, "signer 不能为空");
        final HttpClientBuilder newHttpClientBuilder = Optional.ofNullable(httpClientBuilder)
                .orElseGet(() -> HttpClients.custom()
                        .disableAuthCaching()
                        .disableCookieManagement()
                        .disableRedirectHandling()
                        .disableAutomaticRetries()
                        .setMaxConnPerRoute(20)
                        .setMaxConnTotal(20));

        newHttpClientBuilder.setUserAgent(DEFAULT_USER_AGENT);
        return newHttpClientBuilder
                //.addInterceptorFirst(new EndpointProcessingHttpRequestInterceptor(endpoint))
                .addInterceptorFirst(new SignerHttpRequestInterceptor(signer))
                .build();
    }


    @Data
    @AllArgsConstructor
    public static class OotpClientImpl implements OotpClient, AutoCloseable {
        private final MessageClient messageClient;
        private final FileClient fileClient;
        private final CloseableHttpClient httpClient;

        @Override
        //@PreDestroy
        public void close() throws Exception {
            if (httpClient != null) {
                log.info("关闭 HttpClient");
                try {
                    httpClient.close();
                } catch (Exception e) {
                    // ignore
                }
            }
        }
    }
}
