package org.opoo.ootp.client.impl;

import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.extern.slf4j.Slf4j;
import org.apache.http.HttpEntity;
import org.apache.http.HttpStatus;
import org.apache.http.StatusLine;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPut;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.InputStreamEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.protocol.BasicHttpContext;
import org.apache.http.util.EntityUtils;
import org.opoo.ootp.client.EntityBody;
import org.opoo.ootp.client.ExsBody;
import org.opoo.ootp.client.ExsCodec;
import org.opoo.ootp.client.ExsMessage;
import org.opoo.ootp.client.ExsMetadata;
import org.opoo.ootp.client.FileClient;
import org.opoo.ootp.client.Metadata;
import org.opoo.ootp.client.OotpException;
import org.opoo.ootp.signer.Signer;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.util.AbstractMap;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

@Slf4j
public class FileClientImpl implements FileClient {
    public static final String FILE_ID_FIELD = "fileId";
    private static final String S3_META_PREFIX = "x-amz-meta-";
    private static final int S3_META_PREFIX_LENGTH = S3_META_PREFIX.length();

    private final URI endpoint;
    private final CloseableHttpClient httpClient;
    private final ObjectMapper objectMapper;
    private ExsCodec codec;

    public FileClientImpl(URI endpoint, CloseableHttpClient httpClient, ObjectMapper objectMapper) {
        this.endpoint = endpoint;
        this.httpClient = httpClient;
        this.objectMapper = objectMapper;
    }

    public ExsCodec getCodec() {
        return codec;
    }

    public void setCodec(ExsCodec codec) {
        this.codec = codec;
    }

    @Override
    public String upload(ExsMessage message, String storage, String pathInfo) throws IOException, OotpException {
        final ExsBody body = message.getBody();
        final ExsMetadata metadata = message.getMetadata();
        Objects.requireNonNull(storage, "storage is required.");
        Objects.requireNonNull(metadata, "metadata is required.");
        Objects.requireNonNull(metadata.getContentType(), "contentType is required.");
        Objects.requireNonNull(body, "file is required.");
        Objects.requireNonNull(metadata.getUserMetadata(Metadata.META_FILE_NAME), "fileName is required.");

        if (STORAGE_FS.equals(storage)) {
            Objects.requireNonNull(metadata.getRepo(), "repo is required.");
        } else if (STORAGE_DEFAULT.equals(storage)) {
            Objects.requireNonNull(metadata.getTo(), "to is required.");
        }

        String path = endpoint.toString() + "/file-api/upload";
        if (pathInfo != null) {
            path += pathInfo;
        }
        path += "?storage=" + storage;

        final HttpEntity entity2 = AbstractMessageClient.toEntity(body, metadata);
        final HttpEntity entity = codec != null ? codec.encode(entity2, metadata) : entity2;

        final HttpPut httpPut = new HttpPut(path);
        httpPut.setEntity(entity);
        metadata.toHeaders(httpPut::setHeader);

        try (final CloseableHttpResponse response = httpClient.execute(httpPut)) {
            final StatusLine statusLine = response.getStatusLine();
            final int statusCode = statusLine.getStatusCode();

            final Map<?,?> map;
            try (final InputStream inputStream = response.getEntity().getContent()) {
                map = objectMapper.readValue(inputStream, Map.class);
            }

            if (statusCode < HttpStatus.SC_OK || statusCode >= HttpStatus.SC_MULTIPLE_CHOICES) {
//                if (map != null && (map.containsKey("error") || map.containsKey("message"))) {
//                    final String error = (String) map.get("error");
//                    final String message2 = (String) map.get("message");
//                    throw new OotpException(message2 != null ? message2 : statusCode + "-" + error, statusCode, error);
//                }
                //EntityUtils.consumeQuietly(response.getEntity());
                final String error = Optional.ofNullable(map).map(m -> m.get("error")).map(s -> " - " + s).orElse("");
                throw new OotpException("文件上传失败：" + statusCode + error, statusCode, statusLine.getReasonPhrase());
            }

            return Optional.ofNullable(map).map(m -> (String) m.get(FILE_ID_FIELD))
                    .orElseThrow(() -> new OotpException("响应内容为空", 500, "ResponseEmptyBody"));
        }
    }

    protected <T> T getFile(String fileId, ResponseHandler<T> responseHandler) throws IOException {
        Objects.requireNonNull(fileId, "fileId is required.");
        final HttpGet httpGet = new HttpGet(endpoint.toString() + "/file-api/file/" + fileId);
        try (final CloseableHttpResponse response = httpClient.execute(httpGet)) {
            final StatusLine statusLine = response.getStatusLine();
            final int statusCode = statusLine.getStatusCode();

            if (statusCode < HttpStatus.SC_OK || statusCode >= HttpStatus.SC_MULTIPLE_CHOICES) {
//                final Map<?,?> map;
//                try (final InputStream inputStream = response.getEntity().getContent()) {
//                    map = objectMapper.readValue(inputStream, Map.class);
//                }
//                if (map != null && (map.containsKey("error") || map.containsKey("message"))) {
//                    final String error = (String) map.get("error");
//                    final String message2 = (String) map.get("message");
//                    throw new OotpException(message2 != null ? message2 : statusCode + "-" + error, statusCode, error);
//                }
                //EntityUtils.consumeQuietly(response.getEntity());
                final String string = EntityUtils.toString(response.getEntity(), StandardCharsets.UTF_8);
                throw new OotpException("文件下载失败：" + statusCode + " - " + statusLine.getReasonPhrase() + "\n" + string, statusCode, statusLine.getReasonPhrase());
            }

            final HttpEntity entity = response.getEntity();
            final ContentType contentType = ContentType.get(entity);

            final Map<String,String> headers = new HashMap<>();
            Arrays.stream(response.getAllHeaders()).forEach(h -> headers.putIfAbsent(h.getName().toLowerCase(), h.getValue()));
            final ExsMetadata metadata = ExsMetadata.fromHeaders(headers);
            metadata.setId(fileId);

            log.debug("ExsMetadata for '{}': {}", fileId, metadata);

            final InputStream content = entity.getContent();
            final InputStream decodeStream = codec != null ? codec.decode(content, metadata) : content;
            return responseHandler.handle(decodeStream, contentType, metadata);
        }
    }

    @Override
    public ExsMessage getFile(String fileId) throws IOException {
//        Objects.requireNonNull(fileId, "fileId is required.");
//        final HttpGet httpGet = new HttpGet(endpoint.toString() + "/file-api/file/" + fileId);
//        try (final CloseableHttpResponse response = httpClient.execute(httpGet)) {
//            final StatusLine statusLine = response.getStatusLine();
//            final int statusCode = statusLine.getStatusCode();
//
//            if (statusCode < HttpStatus.SC_OK || statusCode >= HttpStatus.SC_MULTIPLE_CHOICES) {
////                final Map<?,?> map;
////                try (final InputStream inputStream = response.getEntity().getContent()) {
////                    map = objectMapper.readValue(inputStream, Map.class);
////                }
////                if (map != null && (map.containsKey("error") || map.containsKey("message"))) {
////                    final String error = (String) map.get("error");
////                    final String message2 = (String) map.get("message");
////                    throw new OotpException(message2 != null ? message2 : statusCode + "-" + error, statusCode, error);
////                }
//                EntityUtils.consumeQuietly(response.getEntity());
//                throw new OotpException("文件上传失败 - " + statusCode + " - " + statusLine.getReasonPhrase(), statusCode, statusLine.getReasonPhrase());
//            }
//
//            final HttpEntity entity = response.getEntity();
//            final ContentType contentType = ContentType.get(entity);
//
//            final Map<String,String> headers = new HashMap<>();
//            Arrays.stream(response.getAllHeaders()).forEach(h -> headers.putIfAbsent(h.getName().toLowerCase(), h.getValue()));
//            final ExsMetadata metadata = ExsMetadata.fromHeaders(headers);
//            metadata.setId(fileId);
//
//            log.debug("ExsMetadata for '{}': {}", fileId, metadata);
//
//            final InputStream content = entity.getContent();
//            final InputStream decodeStream = codec != null ? codec.decode(content, metadata) : content;
//            final FileCachedInputStream ootpFileCachedInputStream = FileCachedInputStream.create(decodeStream, fileId);
//            final long size = Files.size(ootpFileCachedInputStream.getFile());
//            final InputStreamEntity inputStreamEntity = new InputStreamEntity(ootpFileCachedInputStream, size, contentType);
//            final EntityBody entityBody = new EntityBody(inputStreamEntity);
//
//            return new ExsMessage(entityBody, metadata);
//        }

        return getFile(fileId, ((inputStream, contentType, metadata) -> {
            log.debug("构建自动删除的文件缓存输入流: {}", fileId);
            final FileCachedInputStream ootpFileCachedInputStream = FileCachedInputStream.create(inputStream, fileId);
            final long size = Files.size(ootpFileCachedInputStream.getFile());
            final InputStreamEntity inputStreamEntity = new InputStreamEntity(ootpFileCachedInputStream, size, contentType);
            final EntityBody entityBody = new EntityBody(inputStreamEntity);

            return new ExsMessage(entityBody, metadata);
        }));
    }

    @Override
    public InputStream getStream(String fileId) throws IOException {
        log.debug("构建自动删除的文件缓存输入流：{}", fileId);
        return getFile(fileId, (inputStream, contentType, metadata) -> FileCachedInputStream.create(inputStream, fileId));
    }

    @Override
    public void copy(String fileId, Path destination) throws IOException {
        getFile(fileId, (ResponseHandlerWithoutResult) (inputStream, contentType, metadata) -> {
            log.debug("直接将文件复制到文件: {} -> {}", fileId, destination);
            try (InputStream in = inputStream) {
                Files.copy(in, destination, StandardCopyOption.REPLACE_EXISTING);
            }
        });
    }

    @Override
    public void copy(String fileId, OutputStream outputStream) throws IOException {
        getFile(fileId, (ResponseHandlerWithoutResult) (inputStream, contentType, metadata) -> {
            log.debug("直接将文件复制到输出流: {}", fileId);
            try (InputStream in = inputStream) {
                FileClient.copy(in, outputStream);
            }
        });
    }

    protected <T> T getUri(URI uri, ResponseHandler<T> responseHandler) throws IOException {
        Objects.requireNonNull(uri, "uri is required.");
        final HttpGet httpGet = new HttpGet(uri);
        final BasicHttpContext httpContext = new BasicHttpContext();
        httpContext.setAttribute(Signer.ATTR_SIGNER_SKIP, "true");
        try (final CloseableHttpResponse response = httpClient.execute(httpGet, httpContext)) {
            final StatusLine statusLine = response.getStatusLine();
            final int statusCode = statusLine.getStatusCode();

            if (statusCode < HttpStatus.SC_OK || statusCode >= HttpStatus.SC_MULTIPLE_CHOICES) {
                //EntityUtils.consumeQuietly(response.getEntity());
                final String string = EntityUtils.toString(response.getEntity(), StandardCharsets.UTF_8);
                throw new OotpException("文件下载失败：" + statusCode + " - " + statusLine.getReasonPhrase() + "\n" + string, statusCode, statusLine.getReasonPhrase());
            }

            // 读取 Meta 数据
            final Map<String,String> headers = new HashMap<>();
            Arrays.stream(response.getAllHeaders())
                    .map(h -> new AbstractMap.SimpleEntry<>(h.getName().toLowerCase(), h.getValue()))
                    .filter(e-> e.getKey().startsWith(S3_META_PREFIX))
                    .forEach(e -> headers.putIfAbsent(e.getKey().substring(S3_META_PREFIX_LENGTH), e.getValue()));
            final ExsMetadata metadata = ExsMetadata.fromHeaders(headers);

            log.debug("ExsMetadata for uri: {}", metadata);

            // 处理结果
            final HttpEntity entity = response.getEntity();
            final InputStream content = entity.getContent();
            final InputStream decodeStream = codec != null ? codec.decode(content, metadata) : content;
            //return FileCachedInputStream.create(decodeStream);

            return responseHandler.handle(decodeStream, ContentType.get(entity), metadata);
        }
    }

    @Override
    public ExsMessage getUri(URI uri) throws IOException {
        return getUri(uri, ((inputStream, contentType, metadata) -> {
            log.debug("构建自动删除的文件缓存输入流：{}", uri.getHost());
            final FileCachedInputStream ootpFileCachedInputStream = FileCachedInputStream.create(inputStream);
            final long size = Files.size(ootpFileCachedInputStream.getFile());
            final InputStreamEntity inputStreamEntity = new InputStreamEntity(ootpFileCachedInputStream, size, contentType);
            final EntityBody entityBody = new EntityBody(inputStreamEntity);

            return new ExsMessage(entityBody, metadata);
        }));
    }

    @Override
    public InputStream getStream(URI uri) throws IOException {
        log.debug("构建自动删除的文件缓存输入流：{}", uri.getHost());
        return getUri(uri, (inputStream, contentType, metadata) -> FileCachedInputStream.create(inputStream));
    }

    @Override
    public void copy(URI uri, OutputStream outputStream) throws IOException {
        getUri(uri, (ResponseHandlerWithoutResult)(inputStream, contentType, metadata) -> {
            log.debug("直接将URI复制到输出流：{}", uri.getHost());
            try (InputStream in = inputStream) {
                FileClient.copy(in, outputStream);
            }
        });
    }

    @Override
    public void copy(URI uri, Path destination) throws IOException {
        getUri(uri, (ResponseHandlerWithoutResult)(inputStream, contentType, metadata) -> {
            log.debug("直接将URI复制到文件: {} -> {}", uri.getHost(), destination);
            try (InputStream in = inputStream) {
                Files.copy(in, destination, StandardCopyOption.REPLACE_EXISTING);
            }
        });
    }

    @FunctionalInterface
    interface ResponseHandler<T> {
        T handle(InputStream inputStream, ContentType contentType, ExsMetadata metadata) throws IOException;
    }

    @FunctionalInterface
    interface ResponseHandlerWithoutResult extends ResponseHandler<Object> {
        @Override
        default Object handle(InputStream inputStream, ContentType contentType, ExsMetadata metadata) throws IOException {
            handleWithoutResult(inputStream, contentType, metadata);
            return null;
        }

        void handleWithoutResult(InputStream inputStream, ContentType contentType, ExsMetadata metadata) throws IOException;
    }
}
