package org.opoo.ootp.client;

import org.apache.http.entity.ByteArrayEntity;
import org.apache.http.util.CharArrayBuffer;
import org.apache.http.util.EntityUtils;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.List;
import java.util.function.Consumer;

public interface MessageClient {
    /**
     * 默认上限。
     */
    int DEFAULT_LIMIT = 100;

    int DEFAULT_BUFFER_SIZE = 4096;

    /**
     * 发送消息。
     * @param message 消息
     * @return 机构外联返回的消息ID
     * @throws OotpException 通常是 Http 请求异常，或者服务器返回错误消息
     */
    String send(ExsMessage message) throws OotpException;

    /**
     * 发送消息。
     * @param body 消息体
     * @param metadata 消息元数据
     * @return 机构外联返回的消息ID
     * @throws OotpException 通常是 HTTP 请求异常，或者服务器返回的错误消息
     */
    default String send(ExsBody body, ExsMetadata metadata) throws OotpException {
        return send(new ExsMessage(body, metadata));
    }

    /**
     * 发送消息。
     *
     * @param body 消息体
     * @param metadata 消息元数据
     * @return 机构外联返回的消息ID
     * @throws OotpException 通常是 Http 请求异常，或者服务器返回错误消息
     */
    default String send(String body, ExsMetadata metadata) throws OotpException {
        final StringBody stringBody = new StringBody(body);
        return send(new ExsMessage(stringBody, metadata));
    }

    /**
     * 发送消息。
     * @param type 消息业务类型，用于消息处理方进行标识
     * @param to 接收方
     * @param contentType 消息内容类型
     * @param body 消息体正文
     * @return 服务端返回的消息ID
     * @throws OotpException 通常是 Http 请求异常，或者服务器返回错误消息
     */
    default String send(String type, String to, String contentType, String body) throws OotpException {
        final ExsMetadata metadata = new ExsMetadata();
        metadata.setTo(to);
        metadata.setType(type);
        metadata.setContentType(contentType);
        return send(body, metadata);
    }

    /**
     * 发送消息。
     * @param type 消息业务类型，用于消息处理方进行标识
     * @param to 接收方
     * @param contentType 消息内容类型
     * @param body 消息体正文
     * @return 服务端返回的消息ID
     * @throws OotpException 通常是 Http 请求异常，或者服务器返回错误消息
     * @deprecated 由于机构外联仅接收文本消息，因此建议直接使用 {@link #send(String, String, String, String)} 方法
     * @see #send(String, ExsMetadata)
     * @see #send(String, String, String, String)
     */
    @Deprecated
    default String send(String type, String to, String contentType, byte[] body) throws OotpException {
        final ExsMetadata metadata = new ExsMetadata();
        metadata.setTo(to);
        metadata.setType(type);
        metadata.setContentType(contentType);
        final EntityBody entityBody = new EntityBody(new ByteArrayEntity(body));
        return send(new ExsMessage(entityBody, metadata));
    }

    /**
     * 轮询消息。
     * @param request 轮询请求。
     * @return 轮询结果
     * @throws OotpException 通常是 Http 请求异常，或者服务器返回错误消息
     */
    PollResult poll(PollRequest request) throws OotpException;

    /**
     * 轮询消息。
     * @param limit 每次轮询最大记录数，默认 100
     * @param autoAck 是否自动进行消息确认，即轮询时就发送消息回执，默认 false
     * @param full 是否返回完整的消息记录，默认 false
     * @return 消息集合，可能为空，不为 null
     * @throws OotpException 通常是 Http 请求异常，或者服务器返回错误消息
     */
    default PollResult poll(int limit, boolean autoAck, boolean full) throws OotpException {
        final PollRequest request = new PollRequest(limit, autoAck, full);
        return poll(request);
    }

    /**
     * 使用默认参数轮询消息。
     * @return 消息集合，可能为空，不为 null
     * @throws OotpException 通常是 Http 请求异常，或者服务器返回错误消息
     */
    default PollResult poll() throws OotpException {
        return poll(DEFAULT_LIMIT, false, false);
    }

    /**
     * 获取消息体。默认使用 UTF-8 编码。
     *
     * @param id 消息ID
     * @return 消息体，仅含消息正文
     * @throws OotpException 通常是 Http 请求异常，或者服务器返回错误消息
     */
    default String getMessageBody(String id) throws OotpException {
        final ExsBody body = getMessage(id).getBody();

        if (body instanceof EntityBody) {
            try {
                return EntityUtils.toString(((EntityBody) body).getEntity());
            } catch (IOException e) {
                throw new OotpException("读取消息内容出错", e);
            }
        }

        // return IOUtils.toString(content, StandardCharsets.UTF_8)
        try (final InputStream content = body.getContent();
             final Reader reader = new InputStreamReader(content, StandardCharsets.UTF_8)) {
            final CharArrayBuffer buffer = new CharArrayBuffer(DEFAULT_BUFFER_SIZE);
            final char[] tmp = new char[1024];
            int l;
            while((l = reader.read(tmp)) != -1) {
                buffer.append(tmp, 0, l);
            }
            return buffer.toString();
        } catch (IOException e) {
            throw new OotpException("读取消息内容出错", e);
        }
    }

    /**
     * 获取消息。
     * @param id 消息ID
     * @return 整体消息对象
     * @throws OotpException 通常是 Http 请求异常，或者服务器返回错误消息
     * @since 1.1.0
     */
    ExsMessage getMessage(String id) throws OotpException;

    /**
     * 获取已经发送的消息的信息，如消息的状态，是否被对方消费等。
     * @param id 消息ID
     * @return 消息的信息
     * @throws OotpException 通常是 Http 请求异常，或者服务器返回错误消息
     * @since 1.1.0
     */
    ExsMessageInfo getMessageInfo(String id) throws OotpException;

    /**
     * 确认消息。
     * @param ids 要确认的消息数组，不能为空
     * @return 实际确认的消息数量
     * @throws OotpException 通常是 Http 请求异常，或者服务器返回错误消息
     */
    default int ack(String... ids) throws OotpException {
        return ack(Arrays.asList(ids));
    }

    /**
     * 确认消息。
     * @param ids 要确认的消息集合，不能为空
     * @return 实际确认的消息数量
     * @throws OotpException 通常是 Http 请求异常，或者服务器返回错误消息
     */
    int ack(List<String> ids) throws OotpException;

    /**
     * 退回消息，使消息可以重新消费。已确认超过3天的消息不能退回。
     * @param ids 要确认的消息集合，不能为空
     * @return 实际退回的消息数量
     * @throws OotpException 通常是 Http 请求异常，或者服务器返回错误消息
     * @since 1.1.0
     */
    default int ret(String... ids) throws OotpException {
        return ret(Arrays.asList(ids));
    }

    /**
     * 退回消息，使消息可以重新消费。已确认超过3天的消息不能退回。
     * @param ids 要确认的消息集合，不能为空
     * @return 实际退回的消息数量
     * @throws OotpException 通常是 Http 请求异常，或者服务器返回错误消息
     * @since 1.1.0
     */
    default int ret(List<String> ids) throws OotpException {
        return ret(ids, null);
    }

    /**
     * 退回消息，使消息可以重新消费。已确认超过3天的消息不能退回。
     * @param ids 要确认的消息集合，不能为空
     * @param requestConfigurer 对请求的对象（报文）进行高级配置
     * @return 实际退回的消息数量
     * @throws OotpException 通常是 Http 请求异常，或者服务器返回错误消息
     * @since 1.3.0
     */
    int ret(List<String> ids, Consumer<Object> requestConfigurer) throws OotpException;
}
