package org.opoo.ootp.client.impl;

import org.opoo.ootp.client.ExsMetadata;
import org.opoo.ootp.codec.Codec;
import org.opoo.ootp.codec.binary.BinaryEncodeConstants;
import org.opoo.ootp.codec.encryption.EncryptionUtils;
import org.opoo.ootp.codec.encryption.smx.SMXEncryptionCodec;
import org.opoo.ootp.codec.encryption.smx.SMXEncryptionConstants;

import java.security.PrivateKey;
import java.util.Map;
import java.util.Objects;
import java.util.function.Consumer;

/**
 * 简单的 SM2 + SM4 混合加密解密算法的解码配置器。
 * 当对接方是自己开发的程序进行 SM4 加密、使用 base64 编码、并将加密的 SM4 使用 SM2 加密传输，且没有使用 OOTP 客户端，
 * 也没有将加密/编码的信息通过 metadata 传递过来时，需要根据与对接方的约定，将 metadata
 * 数据进行补全，以便后续解码器能够自动处理。
 * @author alex
 * @see ExsCodecImpl
 */
public class SimpleSMXDecodingMetadataConfigurer implements Consumer<ExsMetadata> {
    /**
     * 常见的 TransformMode 如 SMX:SM4/ECB/NoPadding,BIN:base64; SMX:SM4/CBC/PKCS5Padding,BIN:base64
      */
    private static final String DEFAULT_TRANSFORM_MODE = "SMX:SM4/CBC/PKCS5Padding,BIN:base64";
    private final String keyId;
    private String transformMode = DEFAULT_TRANSFORM_MODE;

    public SimpleSMXDecodingMetadataConfigurer(PrivateKey privateKey) {
        this(EncryptionUtils.getKeyFingerprint(privateKey));
    }

    public SimpleSMXDecodingMetadataConfigurer(String keyId) {
        this.keyId = keyId;
    }

    public String getTransformMode() {
        return transformMode;
    }

    public void setTransformMode(String transformMode) {
        this.transformMode = transformMode;
    }

    @Override
    public void accept(ExsMetadata metadata) {
        final Map<String, String> metaMap = metadata.getUserMetadata();
        metaMap.put(Codec.META_TRANSFORM_MODE, transformMode);
        // meta.put(SMXEncryptionConstants.META_ENCRYPTION_IV, null)
        if (!transformMode.contains(SMXEncryptionCodec.ECB_INDICATOR)) {
            Objects.requireNonNull(metaMap.get(SMXEncryptionConstants.META_ENCRYPTION_IV),
                    "加密算法不是 ECB 填充模式，且元数据中缺少算法的初始化向量（initialization vector, IV): " + transformMode);
        }
        // metaMap.put(SMXEncryptionConstants.META_ENCRYPTION_MESSAGE_KEY, encodedKey)
        Objects.requireNonNull(metaMap.get(SMXEncryptionConstants.META_ENCRYPTION_MESSAGE_KEY),
                "元数据中缺少数据对称加密（SM4）的密钥");
        metaMap.put(SMXEncryptionConstants.META_ENCRYPTION_UNENC_SM3, "");
        metaMap.put(SMXEncryptionConstants.META_ENCRYPTION_UNENC_SIZE, "-1");
        metaMap.put(SMXEncryptionConstants.META_ENCRYPTION_KEY_ID, keyId);
        metaMap.put(BinaryEncodeConstants.META_BIN_UNENCODED_SIZE, "-1");
        metaMap.put(BinaryEncodeConstants.META_BIN_ENCODED_SIZE, "-1");
        metaMap.put(BinaryEncodeConstants.META_BIN_UNENCODED_SM3, "");
    }
}
