package org.opoo.ootp.client.impl;


import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.HttpMessage;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpPut;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.entity.ByteArrayEntity;
import org.apache.http.entity.ContentType;
import org.apache.http.impl.client.CloseableHttpClient;
import org.opoo.ootp.client.ExsBody;
import org.opoo.ootp.client.ExsMessage;
import org.opoo.ootp.client.ExsMessageInfo;
import org.opoo.ootp.client.ExsMessageLite;
import org.opoo.ootp.client.ExsMetadata;
import org.opoo.ootp.client.MessageClient;
import org.opoo.ootp.client.OotpException;
import org.opoo.ootp.client.PollRequest;
import org.opoo.ootp.client.PollResult;
import org.opoo.ootp.signer.Signer;

import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Collectors;

@Slf4j
public class V1MessageClientImpl extends AbstractMessageClient implements MessageClient {

    public V1MessageClientImpl(URI endpoint, CloseableHttpClient httpClient, ObjectMapper objectMapper) {
        this(endpoint, httpClient, objectMapper, null);
    }

    public V1MessageClientImpl(URI endpoint, CloseableHttpClient httpClient, ObjectMapper objectMapper, String basePath) {
        super(endpoint, httpClient, objectMapper, basePath);
    }

    private <T,M extends MessageWrapper> T exec(HttpUriRequest request, Handler<T, M> handler, Class<M> messageWrapperType) {
        processUri(request);
        try (final CloseableHttpResponse response = httpClient.execute(request)) {
            validateResponse(response);

            final M messageWrapper;
            try (final InputStream inputStream = response.getEntity().getContent()) {
                messageWrapper = objectMapper.readValue(inputStream, messageWrapperType);
            }

            if (!messageWrapper.isSuccess()) {
                throw new OotpException(messageWrapper.getErrMsg(), messageWrapper.getCode(), messageWrapper.getErrMsg());
            }

            return handler.apply(response, messageWrapper);
        } catch (IOException ex) {
            throw new OotpException("消息API调用失败：" + ex.getMessage(), ex);
        }
    }

    protected <T> T exec1(HttpUriRequest request, Handler1<T> handler) {
        return exec(request, handler, MessageWrapper1.class);
    }

    protected <T> T exec2(HttpUriRequest request, Handler2<T> handler) {
        return exec(request, handler, MessageWrapper2.class);
    }

    @Override
    public String send(ExsMessage message) throws OotpException {
        final ExsMetadata metadata = Objects.requireNonNull(message.getMetadata(), "metadata is required.");
        Objects.requireNonNull(metadata.getType(), "type is required.");
        Objects.requireNonNull(metadata.getTo(), "to is required.");
        Objects.requireNonNull(metadata.getContentType(), "contentType is required.");
        final ExsBody body = message.getBody();
        Objects.requireNonNull(body, "body is required.");

        HttpPut request = new HttpPut(basePath + "/api/messages/v1");
        try {
            final HttpEntity entity = toEntity(body, metadata);
            request.setEntity(codecEncode(entity, metadata));
        } catch (IOException e) {
            throw new OotpException("处理请求输入时出错", e);
        }
        metadata.toHeaders(request::setHeader);

        return exec1(request, (response, messageWrapper) ->
            Optional.ofNullable(response.getFirstHeader(Signer.HEADER_NAME_PREFIX + "exs-id"))
                    .map(Header::getValue).orElse(null)
        );
    }

    @Override
    public PollResult poll(PollRequest request) throws OotpException {
        final int limit = request.getLimit();
        if (limit <= 0 || limit > PollRequest.DEFAULT_LIMIT) {
            log.warn("修改 poll 请求的消息数量：{} -> {}", limit, DEFAULT_LIMIT);
            request.setLimit(PollRequest.DEFAULT_LIMIT);
        }

        final byte[] bytes;
        try {
            bytes = objectMapper.writeValueAsBytes(request);
        } catch (JsonProcessingException e) {
            throw new OotpException("序列化轮询请求出错", e);
        }

        final HttpPost httpPost = new HttpPost(basePath + "/api/messages/v1");
        httpPost.setEntity(new ByteArrayEntity(bytes, ContentType.APPLICATION_JSON));
        Optional.ofNullable(request.getClientId()).ifPresent(s -> httpPost.setHeader(Signer.HEADER_NAME_PREFIX + "client-id", s));

        return exec1(httpPost, (response, messageWrapper) -> {
            final List<MessageV1> list = messageWrapper.getResultData();

            final PollResult pollResult = new PollResult();

            if (list != null) {
                final List<ExsMessageLite> messages = list.stream().map(m -> {
                    final ExsMessageLite lite = new ExsMessageLite();
                    lite.setBody(m.getMsgBody());

                    final ExsMetadata meta = new ExsMetadata();
                    meta.setId("" + m.getId());
                    meta.setContentType(m.getContentType());
                    meta.setFrom(m.getFromSysId());
                    meta.setType(m.getBizType());
                    lite.setMetadata(meta);
                    return lite;
                }).collect(Collectors.toList());

                pollResult.setMessages(messages);
            }

            Optional.ofNullable(response.getFirstHeader(Signer.HEADER_NAME_PREFIX + "exs-latest-id"))
                    .ifPresent(h -> pollResult.setLatestId(h.getValue()));

            return pollResult;
        });
    }

    @Override
    public ExsMessage getMessage(String id) throws OotpException {
        return getMessage(id, s -> "/api/messages/v1?id=" + s);
//        Objects.requireNonNull(id, "id is required.");
//        final HttpGet httpGet = new HttpGet(basePath + "/api/messages/v1?id=" + id);
//        processUri(httpGet);
//        try (final CloseableHttpResponse response = httpClient.execute(httpGet)) {
//            final HttpEntity entity = response.getEntity();
//            final ContentType contentType = ContentType.get(entity);
//
//            final Map<String,String> headers = new HashMap<>();
//            Arrays.stream(response.getAllHeaders()).forEach(h -> headers.putIfAbsent(h.getName().toLowerCase(), h.getValue()));
//            final ExsMetadata metadata = ExsMetadata.fromHeaders(headers);
//
//            validateResponse(response, () -> {
//                if (contentType == null) {
//                    return false;
//                }
//                final String str = contentType.toString().toLowerCase();
//                // json 报文，且没有编码过，则可以解析
//                return str.startsWith("application/") && str.contains("json") && metadata.getUserMetadata(Codec.META_TRANSFORM_MODE) == null;
//            }, "获取消息[" + id + "]失败");
//
//            final HttpEntity decodedEntity = codecDecode(entity, metadata);
//
////            boolean isJson = false;
////            if (contentType != null) {
////                final String str = contentType.toString().toLowerCase();
////                isJson = str.startsWith("application/") && str.contains("json");
////            }
////
////            final HttpEntity httpEntity;
////            // 如果是 JSON 做二次判断
////            if (isJson && metadata.getUserMetadata(Codec.META_TRANSFORM_MODE) == null) {
////                // 先读出来
////                final byte[] bytes = EntityUtils.toByteArray(decodedEntity);
////
////                final Map<?, ?> map = objectMapper.readValue(bytes, Map.class);
////                Object success = map.get("success");
////                if (success instanceof Boolean && !(boolean) success) {
////                    throw new OotpException("读取消息失败：" + map.get("errMsg"));
////                }
////
////                httpEntity = new ByteArrayEntity(bytes);
////            } else {
////                httpEntity = new BufferedHttpEntity(decodedEntity);
////            }
//
//            final BufferedHttpEntity httpEntity = new BufferedHttpEntity(decodedEntity);
//            return new ExsMessage(new EntityBody(httpEntity), metadata);
//        } catch (IOException ex) {
//            throw new OotpException("获取消息[" + id + "]失败：" + ex.getMessage(), ex);
//        }
    }

    @Override
    public ExsMessageInfo getMessageInfo(String id) throws OotpException {
        Objects.requireNonNull(id, "id is required.");
        final HttpGet httpGet = new HttpGet(basePath + "/api/messages/info/v1?id=" + id);
        return exec2(httpGet, (response, messageWrapper) -> messageWrapper.getResultData());
    }

    @Override
    public int ack(List<String> ids) throws OotpException {
        return processIds(ids, null, "ack");
    }

    @Override
    public int ret(List<String> ids, Consumer<Object> requestConfigurer) throws OotpException {
        return processIds(ids, requestConfigurer, "ret");
    }

    int processIds(List<String> ids, Consumer<Object> requestConfigurer, String action) throws OotpException {
        return processIds(ids, requestConfigurer, "/api/messages/" + action + "/v1", httpPost -> exec2(httpPost, (response, messageWrapper) -> {
            log.debug("{} result: {}", action, messageWrapper);
            return -1;
        }));

//        if (ids == null || ids.isEmpty()) {
//            throw new IllegalArgumentException("ids 不能为空");
//        }
//
//        Map<String, Object> request = new LinkedHashMap<>();
//        request.put("ids", ids);
//
//        final byte[] bytes;
//        try {
//            bytes = objectMapper.writeValueAsBytes(request);
//        } catch (JsonProcessingException e) {
//            throw new OotpException("序列化 " + action + " 请求出错", e);
//        }
//
//        final HttpPost httpPost = new HttpPost(basePath + "/api/messages/" + action + "/v1");
//        httpPost.setEntity(new ByteArrayEntity(bytes, ContentType.APPLICATION_JSON));
//
//        return exec2(httpPost, (response, messageWrapper) -> {
//            log.debug("{} result: {}", action, messageWrapper);
//            return -1;
//        });
    }

    interface MessageWrapper {
        boolean isSuccess();
        int getCode();
        String getErrMsg();
    }

    /**
     * 第一类返回，返回 V1 版消息
     */
    @Data
    public static class MessageWrapper1 implements MessageWrapper {
        private boolean success;
        private int code;
        private String errMsg;
        private List<MessageV1> resultData;
    }

    @Data
    public static class MessageWrapper2 implements MessageWrapper {
        private boolean success;
        private int code;
        private String errMsg;
        private ExsMessageInfo resultData;
    }

    /**
     * V1 版接口的消息
     */
    @Data
    public static class MessageV1 {
        private Long id;
        private String bizType;
        private String fromSysId;
        private String contentType;

        private String msgBody;
    }

    interface Handler<T, M extends MessageWrapper> {
        T apply(HttpMessage response, M messageWrapper) throws IOException;
    }

    @FunctionalInterface
    interface Handler1<T> extends Handler<T,MessageWrapper1> {
        /**
         * 使用 HttpMessage 作为参数即不希望实现类再次处理 entity。
         * @param response 使用 HttpMessage 作为参数即不希望实现类再次处理 entity。
         * @param messageWrapper 返回对象
         * @return 处理后的值
         * @throws IOException IO异常
         */
        @Override
        T apply(HttpMessage response, MessageWrapper1 messageWrapper) throws IOException;
    }

    @FunctionalInterface
    interface Handler2<T> extends Handler<T,MessageWrapper2> {
        @Override
        T apply(HttpMessage response, MessageWrapper2 messageWrapper) throws IOException;
    }
}
