package org.opoo.ootp.client.impl;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.util.VersionInfo;
import org.opoo.ootp.client.ExsCodec;
import org.opoo.ootp.client.FileClient;
import org.opoo.ootp.client.MessageClient;
import org.opoo.ootp.client.OotpClient;
import org.opoo.ootp.signer.AbstractSigner;
import org.opoo.ootp.signer.Signer;
import org.opoo.ootp.signer.httpclient.SignerHttpRequestInterceptor;

import java.net.URI;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

import static com.fasterxml.jackson.databind.DeserializationFeature.FAIL_ON_IGNORED_PROPERTIES;
import static com.fasterxml.jackson.databind.DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES;

@Slf4j
public class OotpClientBuilder {
    public static final String HTTP_VERSION = VersionInfo.getUserAgent("Apache-HttpClient", "org.apache.http.client", HttpClientBuilder.class);
    public static final String PACKAGE_VERSION = OotpClientBuilder.class.getPackage().getImplementationVersion();
    public static final String DEFAULT_USER_AGENT = String.format("Ootp Java Client%s %s (%s; %s; %s)",
            (PACKAGE_VERSION != null ? " v" + PACKAGE_VERSION : ""), HTTP_VERSION,
            System.getProperty("os.name"), System.getProperty("os.version"), System.getProperty("os.arch"));

    private URI endpoint;
    private String systemId;
    private Signer signer;
    private ObjectMapper objectMapper;
    private boolean version1 = false;
    private String basePath;
    private String userAgent;
    private ExsCodec messageCodec;
    private ExsCodec fileCodec;

    private final List<HttpClientConfigurer> httpClientConfigurers = new ArrayList<>();
    private final List<HttpClientConfigurer.RequestConfigConfigurer> requestConfigConfigurers = new ArrayList<>();

    private OotpClientBuilder() {
    }

    public static OotpClient defaultClient(URI endpoint, Signer signer) {
        return new OotpClientBuilder().endpoint(endpoint).signer(signer).build();
    }

    public static OotpClient defaultClient(String endpoint, Signer signer) {
        return defaultClient(URI.create(endpoint), signer);
    }

    public static OotpClientBuilder custom() {
        return new OotpClientBuilder();
    }

    public OotpClientBuilder signer(Signer signer) {
        this.signer = signer;
        return this;
    }

    public OotpClientBuilder endpoint(URI endpoint) {
        this.endpoint = endpoint;
        return this;
    }

    public OotpClientBuilder endpoint(String endpoint) {
        this.endpoint = URI.create(endpoint);
        return this;
    }

    public OotpClientBuilder basePath(String basePath) {
        this.basePath = basePath;
        return this;
    }

    public OotpClientBuilder systemId(String systemId) {
        this.systemId = systemId;
        return this;
    }

    public OotpClientBuilder objectMapper(ObjectMapper objectMapper) {
        this.objectMapper = objectMapper;
        return this;
    }

    public OotpClientBuilder useVersion1() {
        this.version1 = true;
        return this;
    }

    public OotpClientBuilder userAgent(String userAgent) {
        this.userAgent = userAgent;
        return this;
    }

    public OotpClientBuilder messageCodec(ExsCodec codec) {
        this.messageCodec = codec;
        return this;
    }

    public OotpClientBuilder fileCodec(ExsCodec codec) {
        this.fileCodec = codec;
        return this;
    }

    public OotpClientBuilder addHttpClientConfigurer(HttpClientConfigurer configurer) {
        this.httpClientConfigurers.add(configurer);
        return this;
    }

    public OotpClientBuilder httpClientConfigurers(List<HttpClientConfigurer> configurers) {
        this.httpClientConfigurers.clear();
        if (configurers != null) {
            this.httpClientConfigurers.addAll(configurers);
        }
        return this;
    }

    public OotpClientBuilder addHttpClientRequestConfigConfigurer(HttpClientConfigurer.RequestConfigConfigurer requestConfigConfigurer) {
        this.requestConfigConfigurers.add(requestConfigConfigurer);
        return this;
    }

    public OotpClientBuilder httpClientRequestConfigConfigurers(List<HttpClientConfigurer.RequestConfigConfigurer> requestConfigConfigurers) {
        this.requestConfigConfigurers.clear();
        if (requestConfigConfigurers != null) {
            this.requestConfigConfigurers.addAll(requestConfigConfigurers);
        }
        return this;
    }

    /**
     * 创建 HttpClient，由于有专用的拦截器，所有必须是自己构建。
     * @return CloseableHttpClient
     */
    private CloseableHttpClient buildCloseableHttpClient() {
        final HttpClientBuilder httpClientBuilder = HttpClients.custom()
                .disableAuthCaching()
                .disableCookieManagement()
                .disableRedirectHandling()
                .disableAutomaticRetries()
                .setMaxConnPerRoute(20)
                .setMaxConnTotal(20);
        // 个性化配置
        httpClientConfigurers.forEach(c -> c.configure(httpClientBuilder));

        final RequestConfig.Builder builder = RequestConfig.custom()
                .setConnectionRequestTimeout(3000)
                .setConnectTimeout(5000)
                .setSocketTimeout(30000);
        requestConfigConfigurers.forEach(c -> c.configure(builder));
        httpClientBuilder.setDefaultRequestConfig(builder.build());

        return httpClientBuilder
                .setUserAgent(buildUserAgent())
                .addInterceptorFirst(new SignerHttpRequestInterceptor(signer))
                .build();
    }

    private String buildUserAgent() {
        if (userAgent != null) {
            return userAgent;
        }

        if (systemId != null) {
            return DEFAULT_USER_AGENT + " (" + systemId + ")";
        }

        if (signer instanceof AbstractSigner) {
            final String accessKey = ((AbstractSigner) signer).getAccessKey();
            return DEFAULT_USER_AGENT + " (" + accessKey + ")";
        }

        return DEFAULT_USER_AGENT;
    }

    public OotpClient build() {
        Objects.requireNonNull(endpoint, "必须设置 endpoint");
        Objects.requireNonNull(signer, "signer 不能为空");

        final CloseableHttpClient httpClient = buildCloseableHttpClient();

        final ObjectMapper objectMapper = Optional.ofNullable(this.objectMapper)
                .orElseGet(() -> new ObjectMapper()
                        .disable(FAIL_ON_IGNORED_PROPERTIES)
                        .disable(FAIL_ON_UNKNOWN_PROPERTIES)
                        .disable(SerializationFeature.FAIL_ON_EMPTY_BEANS));

        final AbstractMessageClient messageClient = version1 ?
                new V1MessageClientImpl(endpoint, httpClient, objectMapper, basePath) :
                new V2MessageClientImpl(endpoint, httpClient, objectMapper, basePath);

        final FileClientImpl fileClient = new FileClientImpl(endpoint, httpClient, objectMapper);

        Optional.ofNullable(messageCodec).ifPresent(messageClient::setCodec);
        Optional.ofNullable(fileCodec).ifPresent(fileClient::setCodec);

        // 相同的  httpclient 则表明是外部注入的，不是自己生成不要自己关闭
        return new OotpClientImpl(messageClient, fileClient, httpClient);
    }

    @Data
    @AllArgsConstructor
    static class OotpClientImpl implements OotpClient, AutoCloseable {
        private final MessageClient messageClient;
        private final FileClient fileClient;
        private final CloseableHttpClient httpClient;

        @Override
        public void close() throws Exception {
            if (httpClient != null) {
                log.info("关闭 HttpClient");
                try {
                    httpClient.close();
                } catch (Exception e) {
                    // ignore
                }
            }
        }
    }

}
