package org.opoo.ootp.client.impl;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.apache.http.HttpEntity;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpPut;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.entity.ByteArrayEntity;
import org.apache.http.entity.ContentType;
import org.apache.http.impl.client.CloseableHttpClient;
import org.opoo.ootp.client.ExsBody;
import org.opoo.ootp.client.ExsMessage;
import org.opoo.ootp.client.ExsMessageInfo;
import org.opoo.ootp.client.ExsMessageLite;
import org.opoo.ootp.client.ExsMetadata;
import org.opoo.ootp.client.MessageClient;
import org.opoo.ootp.client.OotpException;
import org.opoo.ootp.client.PollRequest;
import org.opoo.ootp.client.PollResult;
import org.opoo.ootp.signer.Signer;

import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Collectors;

@Slf4j
public class V2MessageClientImpl extends AbstractMessageClient implements MessageClient {

    public V2MessageClientImpl(URI endpoint, CloseableHttpClient httpClient, ObjectMapper objectMapper) {
        this(endpoint, httpClient, objectMapper, null);
    }

    public V2MessageClientImpl(URI endpoint, CloseableHttpClient httpClient, ObjectMapper objectMapper, String basePath) {
        super(endpoint, httpClient, objectMapper, basePath);
    }

    protected <T> T exec(HttpUriRequest request, Handler<T> handler) {
        processUri(request);
        try (final CloseableHttpResponse response = httpClient.execute(request)) {
            validateResponse(response);

            return handler.apply(response, response.getEntity());
        } catch (IOException ex) {
            throw new OotpException("消息API调用失败：" + ex.getMessage(), ex);
        }
    }

    @Override
    public String send(ExsMessage message) throws OotpException {
        final ExsMetadata metadata = Objects.requireNonNull(message.getMetadata(), "metadata is required.");
        Objects.requireNonNull(metadata.getType(), "type is required.");
        Objects.requireNonNull(metadata.getTo(), "to is required.");
        Objects.requireNonNull(metadata.getContentType(), "contentType is required.");
        final ExsBody body = message.getBody();
        Objects.requireNonNull(body, "body is required.");

        HttpPut request = new HttpPut(basePath + "/api/messages/v2");
        try {
            final HttpEntity entity = toEntity(body, metadata);
            request.setEntity(codecEncode(entity, metadata));
        } catch (IOException e) {
            throw new OotpException("处理请求输入时出错", e);
        }

        processCjkExtensionHeader(request, metadata);
        metadata.toHeaders(request::setHeader);

        return exec(request, (response, entity) -> {
            try (final InputStream inputStream = entity.getContent()) {
                final Map<?, ?> map = objectMapper.readValue(inputStream, Map.class);
                return (String) map.get("id");
            }
        });
    }

    @Override
    public PollResult poll(PollRequest request) throws OotpException {
        final int limit = request.getLimit();
        if (limit <= 0 || limit > PollRequest.DEFAULT_LIMIT) {
            log.warn("修改 poll 请求的消息数量：{} -> {}", limit, DEFAULT_LIMIT);
            request.setLimit(PollRequest.DEFAULT_LIMIT);
        }

        final byte[] bytes;
        try {
            bytes = objectMapper.writeValueAsBytes(request);
        } catch (JsonProcessingException e) {
            throw new OotpException("序列化轮询请求出错", e);
        }

        final HttpPost httpPost = new HttpPost(basePath + "/api/messages/v2");
        httpPost.setEntity(new ByteArrayEntity(bytes, ContentType.APPLICATION_JSON));
        Optional.ofNullable(request.getClientId()).ifPresent(s -> httpPost.setHeader(Signer.HEADER_NAME_PREFIX + "client-id", s));

        return exec(httpPost, ((response, entity) -> {
            final PollResult pollResult = new PollResult();

            final MessageV2[] arr;
            try (final InputStream inputStream = entity.getContent()) {
                arr = objectMapper.readValue(inputStream, MessageV2[].class);
            }

            if (arr != null) {
                final List<ExsMessageLite> messages = Arrays.stream(arr).map(m -> {
                    final ExsMessageLite lite = new ExsMessageLite();
                    lite.setBody(m.getBody());

                    final ExsMetadata meta = new ExsMetadata();
                    meta.setId(m.getId());
                    meta.setFrom(m.getFrom());
                    meta.setType(m.getType());
                    meta.setContentType(m.getContentType());
                    lite.setMetadata(meta);
                    return lite;
                }).collect(Collectors.toList());

                pollResult.setMessages(messages);
            }

            Optional.ofNullable(response.getFirstHeader(Signer.HEADER_NAME_PREFIX + "exs-latest-id"))
                    .ifPresent(h -> pollResult.setLatestId(h.getValue()));

            return pollResult;

        }));
    }

    @Override
    public ExsMessage getMessage(String id) throws OotpException {
        return getMessage(id, s -> "/api/messages/v2?id=" + s);
//        Objects.requireNonNull(id, "id is required.");
//        final HttpGet httpGet = new HttpGet(basePath + "/api/messages/v2?id=" + id);
//        processUri(httpGet);
//        try (final CloseableHttpResponse response = httpClient.execute(httpGet)) {
//            final StatusLine statusLine = response.getStatusLine();
//            final HttpEntity entity = response.getEntity();
//            final ContentType contentType = ContentType.get(entity);
//            final int statusCode = statusLine.getStatusCode();
//
//            final Map<String,String> headers = new HashMap<>();
//            Arrays.stream(response.getAllHeaders()).forEach(h -> headers.putIfAbsent(h.getName().toLowerCase(), h.getValue()));
//            final ExsMetadata metadata = ExsMetadata.fromHeaders(headers);
//
//            validateResponse(response, () -> {
//                if (contentType == null) {
//                    return false;
//                }
//                final String str = contentType.toString().toLowerCase();
//                // json 报文，且没有编码过，则可以解析
//                return str.startsWith("application/") && str.contains("json") && metadata.getUserMetadata(Codec.META_TRANSFORM_MODE) == null;
//            }, "获取消息[" + id + "]失败");
//
//            final HttpEntity decodedEntity = codecDecode(entity, metadata);
//
//            final BufferedHttpEntity httpEntity = new BufferedHttpEntity(decodedEntity);
//            return new ExsMessage(new EntityBody(httpEntity), metadata);
//        } catch (IOException ex) {
//            throw new OotpException("获取消息[" + id + "]失败：" + ex.getMessage(), ex);
//        }
    }

    @Override
    public ExsMessageInfo getMessageInfo(String id) throws OotpException {
        Objects.requireNonNull(id, "id is required.");
        final HttpGet httpGet = new HttpGet(basePath + "/api/messages/info/v2?id=" + id);
        return exec(httpGet, (response, entity) -> {
            try (final InputStream inputStream = entity.getContent()) {
                return objectMapper.readValue(inputStream, ExsMessageInfo.class);
            }
        });
    }

    @Override
    public int ack(List<String> ids) throws OotpException {
        return processIds(ids, null, "ack");
    }

    @Override
    public int ret(List<String> ids, Consumer<Object> requestConfigurer) throws OotpException {
        return processIds(ids, requestConfigurer, "ret");
    }

    int processIds(List<String> ids, Consumer<Object> requestConfigurer, String action) throws OotpException {
        return processIds(ids, requestConfigurer, "/api/messages/" + action + "/v2", httpPost -> exec(httpPost, (response, entity) -> {
            final Map<?, ?> result;
            try (final InputStream inputStream = entity.getContent()) {
                result = objectMapper.readValue(inputStream, Map.class);
            }
            log.debug("{} result: {}", action, result);

            return ((Number) result.get("count")).intValue();
        }));

//        if (ids == null || ids.isEmpty()) {
//            throw new IllegalArgumentException("ids 不能为空");
//        }
//
//        Map<String, Object> request = new LinkedHashMap<>();
//        request.put("ids", ids);
//
//        final byte[] bytes;
//        try {
//            bytes = objectMapper.writeValueAsBytes(request);
//        } catch (JsonProcessingException e) {
//            throw new OotpException("序列化 " + action + " 请求出错", e);
//        }
//
//        final HttpPost httpPost = new HttpPost(basePath + "/api/messages/" + action + "/v2");
//        httpPost.setEntity(new ByteArrayEntity(bytes, ContentType.APPLICATION_JSON));
//
//        return exec(httpPost, (response, entity) -> {
//            final Map<?, ?> result;
//            try (final InputStream inputStream = entity.getContent()) {
//                result = objectMapper.readValue(inputStream, Map.class);
//            }
//            log.debug("{} result: {}", action, result);
//
//            return ((Number) result.get("count")).intValue();
//        });
    }

    /**
     * V1 版接口的消息
     */
    @Data
    public static class MessageV2 {
        private String id;
        private String type;
        private String from;
        private String contentType;

        private String body;
    }

    @FunctionalInterface
    interface Handler<T> {
        T apply(CloseableHttpResponse response, HttpEntity entity) throws IOException;
    }
}
