package org.opoo.ootp.client;

import org.apache.http.entity.ContentType;
import org.apache.http.entity.FileEntity;
import org.apache.http.entity.InputStreamEntity;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URI;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.util.Date;
import java.util.Objects;

public interface FileClient {
    int DEFAULT_BUFFER_SIZE = 4096;

    String STORAGE_DEFAULT = "default";
    String STORAGE_FS = "fs";

    /**
     * 上传文件。向机构外联数据传输平台发送文件。
     * @param message 文件消息
     * @param storage 存储器名称
     * @param pathInfo 后置处理路径
     * @return 上传后生成的文件ID等信息
     * @throws IOException 文件处理异常
     * @throws OotpException 通常是 Http 请求异常，或者服务器返回错误消息
     * @since 1.4
     */
    FileInfo put(ExsMessage message, String storage, String pathInfo) throws IOException, OotpException;

    /**
     * 上传文件。
     * @param message 文件消息
     * @param storage 存储器名称
     * @param pathInfo 后置处理路径
     * @return 上传后生成的文件ID
     * @throws IOException 文件处理异常
     * @throws OotpException 通常是 Http 请求异常，或者服务器返回错误消息
     */
    default String upload(ExsMessage message, String storage, String pathInfo) throws IOException, OotpException {
        return put(message, storage, pathInfo).getFileId();
    }

    /**
     * 上传文件。
     * @param message 文件消息
     * @param storage 存储器名称
     * @return 上传后生成的文件ID
     * @throws IOException 文件处理异常
     * @throws OotpException 通常是 Http 请求异常，或者服务器返回错误消息
     */
    default String upload(ExsMessage message, String storage) throws IOException, OotpException {
        return upload(message, storage, null);
    }

    /**
     * 上传文件。
     * @param message 文件消息
     * @return 上传后生成的文件ID
     * @throws IOException 文件处理异常
     * @throws OotpException 通常是 Http 请求异常，或者服务器返回错误消息
     */
    default String upload(ExsMessage message) throws IOException, OotpException {
        return upload(message, STORAGE_DEFAULT, null);
    }

    /**
     * 上传文件
     * @param stream 文件内容
     * @param metadata 元数据
     * @param storage 存储器
     * @param pathInfo 后置处理路径
     * @return 文件ID
     * @throws IOException 文件处理异常
     * @throws OotpException 通常是 Http 请求异常，或者服务器返回错误消息
     */
    default String upload(InputStream stream, ExsMetadata metadata, String storage, String pathInfo) throws IOException, OotpException {
        Objects.requireNonNull(metadata, "metadata is required.");
        Objects.requireNonNull(metadata.getContentType(), "contentType is required.");

        final String contentType = metadata.getContentType();
        final Long contentLength = metadata.getContentLength();

        final InputStreamEntity entity;
        if (contentLength != null && contentLength > 0) {
            entity = new InputStreamEntity(stream, contentLength, ContentType.parse(contentType));
        } else {
            entity = new InputStreamEntity(stream, ContentType.parse(contentType));
        }

        return upload(new ExsMessage(new EntityBody(entity), metadata), storage, pathInfo);
    }

    /**
     * 上传文件
     * @param stream 文件内容
     * @param metadata 元数据
     * @param storage 存储器
     * @return 文件ID
     * @throws IOException 文件处理异常
     * @throws OotpException 通常是 Http 请求异常，或者服务器返回错误消息
     */
    default String upload(InputStream stream, ExsMetadata metadata, String storage) throws IOException, OotpException {
        return upload(stream, metadata, storage, null);
    }

    /**
     * 上传文件
     * @param stream 文件内容
     * @param metadata 元数据
     * @return 文件ID
     * @throws IOException 文件处理异常
     * @throws OotpException 通常是 Http 请求异常，或者服务器返回错误消息
     */
    default String upload(InputStream stream, ExsMetadata metadata) throws IOException, OotpException {
        return upload(stream, metadata, STORAGE_DEFAULT, null);
    }

    /**
     * 上传文件到文件服务文件库。
     *
     * @param repo 目标文件库
     * @param contentType 文件内容类型
     * @param file 文件
     * @return 上传后返回的文件ID
     * @throws IOException 文件处理异常
     * @throws OotpException 通常是 Http 请求异常，或者服务器返回错误消息
     */
    default String uploadToFsStorage(String repo, String contentType, File file) throws IOException, OotpException {
        return uploadToFsStorage(repo, contentType, file, null);
    }

    /**
     * 上传文件到文件服务文件库。
     *
     * @param repo 目标文件库
     * @param contentType 文件内容类型
     * @param file 文件
     * @param pathInfo 文件上传的后置处理
     * @return 上传后返回的文件ID
     * @throws IOException 文件处理异常
     * @throws OotpException 通常是 Http 请求异常，或者服务器返回错误消息
     */
    default String uploadToFsStorage(String repo, String contentType, File file, String pathInfo) throws IOException, OotpException {
        Objects.requireNonNull(file, "file is required.");

        final ExsMetadata exsMetadata = new ExsMetadata().withRepo(repo)
                .withContentType(contentType)
                .withContentLength(file.length())
                .addEncodedUserMetadata(Metadata.META_FILE_NAME, file.getName());

        final long lastModified = file.lastModified();
        if (lastModified > 0) {
            //exsMetadata.addUserMetadata(Metadata.META_LAST_MODIFIED, DateUtils.formatDate(new Date(lastModified)));
            exsMetadata.setLastModified(new Date(lastModified));
        }

        final FileEntity fileEntity = new FileEntity(file, ContentType.parse(contentType));
        final ExsMessage exsMessage = new ExsMessage(new EntityBody(fileEntity), exsMetadata);
        return upload(exsMessage, STORAGE_FS, pathInfo);
    }

    /**
     * 上传文件到文件服务文件库。
     *
     * @param repo 目标文件库
     * @param contentType 文件内容类型
     * @param file 文件内容
     * @param fileName 文件名
     * @param fileSize 文件大小，未知大小传 -1
     * @param lastModified 文件最后更新时间
     * @return 上传后返回的文件ID
     * @throws IOException 文件处理异常
     * @throws OotpException 通常是 Http 请求异常，或者服务器返回错误消息
     */
    default String uploadToFsStorage(String repo, String contentType, InputStream file, String fileName, long fileSize, long lastModified)
            throws IOException, OotpException {
        return uploadToFsStorage(repo, contentType, file, fileName, fileSize, lastModified, null);
    }

    /**
     * 上传文件到文件服务文件库。
     *
     * @param repo 目标文件库
     * @param contentType 文件内容类型
     * @param file 文件内容
     * @param fileName 文件名
     * @param fileSize 文件大小，未知大小传 -1
     * @param lastModified 文件最后更新时间
     * @param pathInfo 文件上传的后置处理
     * @return 上传后返回的文件ID
     * @throws IOException 文件处理异常
     * @throws OotpException 通常是 Http 请求异常，或者服务器返回错误消息
     */
    default String uploadToFsStorage(String repo, String contentType, InputStream file, String fileName, long fileSize, long lastModified, String pathInfo)
            throws IOException, OotpException {
        Objects.requireNonNull(file, "file is required.");

        final ExsMetadata exsMetadata = new ExsMetadata().withRepo(repo)
                .withContentType(contentType)
                .withContentLength(fileSize)
                .addEncodedUserMetadata(Metadata.META_FILE_NAME, fileName);

        if (lastModified > 0) {
            //exsMetadata.addUserMetadata(Metadata.META_LAST_MODIFIED, DateUtils.formatDate(new Date(lastModified)));
            exsMetadata.setLastModified(new Date(lastModified));
        }

        return upload(file, exsMetadata, STORAGE_FS, pathInfo);
    }

    /**
     * 上传文件到默认文件中转服务。
     * @param to 目标接收方
     * @param contentType 文件内容类型
     * @param file 文件
     * @return 上传后返回的文件ID
     * @throws IOException 文件处理异常
     * @throws OotpException 其它异常
     */
    default String uploadToDefaultStorage(String to, String contentType, File file) throws IOException, OotpException {
        return upload(STORAGE_DEFAULT, to, contentType, file);
    }

    /**
     * 上传文件到默认文件中转服务。
     * @param to 目标接收方
     * @param contentType 文件内容类型
     * @param file 文件
     * @param pathInfo 文件上传的后置处理
     * @return 上传后返回的文件ID
     * @throws IOException 文件处理异常
     * @throws OotpException 其它异常
     */
    default String uploadToDefaultStorage(String to, String contentType, File file, String pathInfo) throws IOException, OotpException {
        return upload(STORAGE_DEFAULT, to, contentType, file, pathInfo);
    }

    /**
     * 上传文件到默认文件中转服务。
     *
     * @param to 目标接收方
     * @param contentType 文件内容类型
     * @param file 文件内容
     * @param fileName 文件名
     * @param fileSize 文件大小，未知大小传 -1
     * @param lastModified 文件最后更新时间
     * @return 上传后返回的文件ID
     * @throws IOException 文件处理异常
     * @throws OotpException 其它异常
     */
    default String uploadToDefaultStorage(String to, String contentType, InputStream file, String fileName, long fileSize, long lastModified)
            throws IOException, OotpException {
        return upload(STORAGE_DEFAULT, to, contentType, file, fileName, fileSize, lastModified);
    }

    /**
     * 上传文件到默认文件中转服务。
     *
     * @param to 目标接收方
     * @param contentType 文件内容类型
     * @param file 文件内容
     * @param fileName 文件名
     * @param fileSize 文件大小，未知大小传 -1
     * @param lastModified 文件最后更新时间
     * @param pathInfo 文件上传的后置处理
     * @return 上传后返回的文件ID
     * @throws IOException 文件处理异常
     * @throws OotpException 其它异常
     */
    default String uploadToDefaultStorage(String to, String contentType, InputStream file, String fileName, long fileSize, long lastModified, String pathInfo)
            throws IOException, OotpException {
        return upload(STORAGE_DEFAULT, to, contentType, file, fileName, fileSize, lastModified, pathInfo);
    }

    /**
     * 上传到个性化的存储器中。
     *
     * @param storage 存储器名称
     * @param contentType 文件内容类型
     * @param file 文件
     * @return 上传后返回的文件ID
     * @throws IOException 文件处理异常
     * @throws OotpException 其它异常
     */
    default String uploadToCustomStorage(String storage, String contentType, File file) throws IOException, OotpException {
        return upload(storage, null, contentType, file);
    }

    /**
     * 上传到个性化的存储器中。
     *
     * @param storage 存储器名称
     * @param contentType 文件内容类型
     * @param file 文件
     * @param pathInfo 文件上传的后置处理
     * @return 上传后返回的文件ID
     * @throws IOException 文件处理异常
     * @throws OotpException 其它异常
     */
    default String uploadToCustomStorage(String storage, String contentType, File file, String pathInfo) throws IOException, OotpException {
        return upload(storage, null, contentType, file, pathInfo);
    }

    /**
     * 上传到个性化的存储器中。
     * @param storage 存储器名称
     * @param contentType 文件内容类型
     * @param file 文件内容
     * @param fileName 文件名
     * @param fileSize 文件大小，未知大小传 -1
     * @param lastModified 文件最后更新时间
     * @return 上传后返回的文件ID
     * @throws IOException 文件处理异常
     * @throws OotpException 其它异常
     */
    default String uploadToCustomStorage(String storage, String contentType, InputStream file, String fileName, long fileSize, long lastModified)
            throws IOException, OotpException {
        return upload(storage, null, contentType, file, fileName, fileSize, lastModified);
    }

    /**
     * 上传到个性化的存储器中。
     * @param storage 存储器名称
     * @param contentType 文件内容类型
     * @param file 文件内容
     * @param fileName 文件名
     * @param fileSize 文件大小，未知大小传 -1
     * @param lastModified 文件最后更新时间
     * @param pathInfo 文件上传的后置处理
     * @return 上传后返回的文件ID
     * @throws IOException 文件处理异常
     * @throws OotpException 其它异常
     */
    default String uploadToCustomStorage(String storage, String contentType, InputStream file, String fileName, long fileSize, long lastModified, String pathInfo)
            throws IOException, OotpException {
        return upload(storage, null, contentType, file, fileName, fileSize, lastModified, pathInfo);
    }

    /**
     * 上传文件。
     * @param storage 存储器的名称
     * @param to 目标接收方
     * @param contentType 文件内容类型
     * @param file 文件
     * @return 上传后返回的文件ID
     * @throws IOException 文件处理异常
     * @throws OotpException 其它异常
     */
    default String upload(String storage, String to, String contentType, File file) throws IOException, OotpException {
        return upload(storage, to, contentType, file, null);
    }

    /**
     * 上传文件。
     * @param storage 存储器的名称
     * @param to 目标接收方
     * @param contentType 文件内容类型
     * @param file 文件
     * @param pathInfo 文件上传的后置处理
     * @return 上传后返回的文件ID
     * @throws IOException 文件处理异常
     * @throws OotpException 其它异常
     */
    default String upload(String storage, String to, String contentType, File file, String pathInfo) throws IOException, OotpException {
        /*
        try (FileInputStream fis = new FileInputStream(file)) {
            return upload(storage, to, contentType, fis, file.getName(), file.length(), file.lastModified(), pathInfo);
        }
        */

        final ExsMetadata exsMetadata = new ExsMetadata().withTo(to)
                .withContentType(contentType)
                .withContentLength(file.length())
                .addEncodedUserMetadata(Metadata.META_FILE_NAME, file.getName());

        final long lastModified = file.lastModified();
        if (lastModified > 0) {
            //exsMetadata.addUserMetadata(Metadata.META_LAST_MODIFIED, DateUtils.formatDate(new Date(lastModified)));
            exsMetadata.setLastModified(new Date(lastModified));
        }

        final FileEntity fileEntity = new FileEntity(file, ContentType.parse(contentType));
        final ExsMessage exsMessage = new ExsMessage(new EntityBody(fileEntity), exsMetadata);
        return upload(exsMessage, storage, pathInfo);
    }

    /**
     * 上传文件。
     *
     * @param storage 存储器的名称
     * @param to 目标接收方，接收方ID或者文件库ID
     * @param contentType 文件内容类型
     * @param file 文件内容
     * @param fileName 文件名
     * @param fileSize 文件大小，未知大小传 -1
     * @param lastModified 文件最后更新时间
     * @return 上传后返回的文件ID
     * @throws IOException 文件处理异常
     * @throws OotpException 其它异常
     */
    default String upload(String storage, String to, String contentType, InputStream file, String fileName, long fileSize, long lastModified)
            throws IOException, OotpException {
        return upload(storage, to, contentType, file, fileName, fileSize, lastModified, null);
    }

    /**
     * 上传文件。
     *
     * @param storage 存储器的名称
     * @param to 目标接收方，接收方ID或者文件库ID
     * @param contentType 文件内容类型
     * @param file 文件内容
     * @param fileName 文件名
     * @param fileSize 文件大小，未知大小传 -1
     * @param lastModified 文件最后更新时间
     * @param pathInfo 文件后置处理路径
     * @return 上传后返回的文件ID
     * @throws IOException 文件处理异常
     * @throws OotpException 其它异常
     */
    default String upload(String storage, String to, String contentType, InputStream file, String fileName, long fileSize, long lastModified, String pathInfo)
            throws IOException, OotpException {
        final ExsMetadata exsMetadata = new ExsMetadata().withTo(to)
                .withContentType(contentType)
                .withContentLength(fileSize)
                .addEncodedUserMetadata(Metadata.META_FILE_NAME, fileName);

        if (lastModified > 0) {
            //exsMetadata.addUserMetadata(Metadata.META_LAST_MODIFIED, DateUtils.formatDate(new Date(lastModified)));
            exsMetadata.setLastModified(new Date(lastModified));
        }

        return upload(file, exsMetadata, storage, pathInfo);
    }

    /**
     * 获取文件及附加信息。
     * @param fileId 文件ID
     * @return 文件及附加信息
     * @throws IOException 文件处理异常
     */
    ExsMessage getFile(String fileId) throws IOException;

    /**
     * 下载文件。仅上传到默认中转服务的文件可以下载。
     *
     * @param fileId 文件ID
     * @return 文件流
     * @throws IOException 文件处理异常
     */
    default InputStream getStream(String fileId) throws IOException {
        return getFile(fileId).getBody().getContent();
    }

    /**
     * 复制到本地文件。仅上传到默认中转服务的文件可以复制。
     * @param fileId 文件ID
     * @param destination 本地文件
     * @throws IOException io 异常
     */
    default void copy(String fileId, File destination) throws IOException {
//        try (InputStream inputStream = download(fileId);
//             ReadableByteChannel rbc = Channels.newChannel(inputStream);
//             FileOutputStream fileOutputStream = new FileOutputStream(destination)) {
//            fileOutputStream.getChannel().transferFrom(rbc, 0, Long.MAX_VALUE);
//        }

        try (FileOutputStream fos = new FileOutputStream(destination)) {
            copy(fileId, fos);
        }
    }

    /**
     * 复制到本地文件。仅上传到默认中转服务的文件可以复制。
     * @param fileId 文件ID
     * @param destination 本地文件
     * @throws IOException io 异常
     */
    default void copy(String fileId, Path destination) throws IOException {
        try (InputStream inputStream = getStream(fileId)) {
            Files.copy(inputStream, destination, StandardCopyOption.REPLACE_EXISTING);
        }
    }

    /**
     * 复制到本地输出。仅上传到默认中转服务的文件可以复制。
     * @param fileId 文件ID
     * @param outputStream 输出流
     * @throws IOException io 异常
     */
    default void copy(String fileId, OutputStream outputStream) throws IOException {
        try (InputStream inputStream = getStream(fileId)) {
            copy(inputStream, outputStream);
        }
    }

    /**
     * 获取 URI 对应的文件的文件信息，如果远程文件是解密的，这里是解密后的文件信息。
     * @param uri 远程文件资源地址。一般是预签名的S3对象的URI地址。
     * @return 解密/解码后的输入流
     * @throws IOException io 异常
     */
    ExsMessage getUri(URI uri) throws IOException;

    /**
     * 获取 URI 对应的文件的输入流，如果远程文件是解密的，这里是解密后的输入流。
     * @param uri 远程文件资源地址。一般是预签名的S3对象的URI地址。
     * @return 解密/解码后的输入流
     * @throws IOException io 异常
     */
    default InputStream getStream(URI uri) throws IOException {
        return getUri(uri).getBody().getContent();
    }

    default void copy(URI uri, OutputStream outputStream) throws IOException {
        try (InputStream inputStream = getStream(uri)) {
            copy(inputStream, outputStream);
        }
    }

    default void copy(URI uri, File destination) throws IOException {
        try (FileOutputStream fos = new FileOutputStream(destination)) {
            copy(uri, fos);
        }
    }

    default void copy(URI uri, Path destination) throws IOException {
        try (InputStream inputStream = getStream(uri)) {
            Files.copy(inputStream, destination, StandardCopyOption.REPLACE_EXISTING);
        }
    }


    /**
     * 将数据从输入流复制到输出流。
     * @param in 输入流
     * @param out 输出流
     * @return 字节数
     * @throws IOException IO异常
     */
    static int copy(InputStream in, OutputStream out) throws IOException {
        Objects.requireNonNull(in, "No InputStream specified");
        Objects.requireNonNull(out, "No OutputStream specified");

        int byteCount = 0;
        byte[] buffer = new byte[DEFAULT_BUFFER_SIZE];
        int bytesRead;
        while ((bytesRead = in.read(buffer)) != -1) {
            out.write(buffer, 0, bytesRead);
            byteCount += bytesRead;
        }
        out.flush();
        return byteCount;
    }

}
