package org.opoo.ootp.client.impl;

import org.opoo.ootp.client.ExsMetadata;
import org.opoo.ootp.codec.Codec;
import org.opoo.ootp.codec.binary.BinaryEncodeConstants;
import org.opoo.ootp.codec.encryption.EncryptionUtils;
import org.opoo.ootp.codec.encryption.KeyAndSpecProvider;
import org.opoo.ootp.codec.encryption.sm4.SM4EncryptionConstants;

import javax.crypto.SecretKey;
import java.util.Map;
import java.util.function.Consumer;

/**
 * 简单的 SM4 对称加密解密算法的解码配置器。
 * 当对接方是自己开发的程序进行 SM4 加密并使用 base64 编码，且没有使用 OOTP 客户端，
 * 也没有将加密/编码的信息通过 metadata 传递过来时，需要根据与对接方的约定，将 metadata
 * 数据进行补全，以便后续解码器能够自动处理。
 * @author alex
 * @see ExsCodecImpl
 */
public class SimpleSM4DecodingMetadataConfigurer implements Consumer<ExsMetadata> {
    private static final String DEFAULT_TRANSFORM_MODE = "SM4:SM4/ECB/PKCS5Padding,BIN:base64";
    private final String keyHash;
    private String transformMode = DEFAULT_TRANSFORM_MODE;

    public SimpleSM4DecodingMetadataConfigurer(SecretKey secretKey) {
        this(EncryptionUtils.getKeyHash(secretKey));
    }

    public SimpleSM4DecodingMetadataConfigurer(String keyHash) {
        this.keyHash = keyHash;
    }

    public SimpleSM4DecodingMetadataConfigurer(KeyAndSpecProvider keyAndSpecProvider) {
        this.keyHash = keyAndSpecProvider.getMasterKeyHash();
    }

    public String getTransformMode() {
        return transformMode;
    }

    public void setTransformMode(String transformMode) {
        this.transformMode = transformMode;
    }

    @Override
    public void accept(ExsMetadata metadata) {
        final Map<String, String> metaMap = metadata.getUserMetadata();
        metaMap.put(Codec.META_TRANSFORM_MODE, transformMode);
        metaMap.put(SM4EncryptionConstants.META_ENCRYPTION_KEY_HASH, keyHash);
        metaMap.put(SM4EncryptionConstants.META_ENCRYPTION_UNENC_SM3, "");
        metaMap.put(SM4EncryptionConstants.META_ENCRYPTION_UNENC_SIZE, "-1");
        metaMap.put(BinaryEncodeConstants.META_BIN_UNENCODED_SIZE, "-1");
        metaMap.put(BinaryEncodeConstants.META_BIN_ENCODED_SIZE, "-1");
        metaMap.put(BinaryEncodeConstants.META_BIN_UNENCODED_SM3, "");
    }
}
