package org.opoo.ootp.client;

import lombok.Data;
import org.apache.http.client.utils.DateUtils;

import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.TreeMap;
import java.util.function.BiConsumer;

import static org.opoo.ootp.signer.Signer.HEADER_NAME_CONTENT_HASH;

/**
 * 消息元数据定义
 */
@Data
public class ExsMetadata implements Metadata {
    private String id;
    private String type;
    private String from;
    private String to;
    private String repo;
    private String contentType;
    private Long contentLength;
    private String contentHash;
    private Date lastModified;

    private Map<String, String> userMetadata = new TreeMap<String, String>(String.CASE_INSENSITIVE_ORDER);

    /**
     * Stores a copy of the map to prevent internal alterations (i.e. codecs) from affecting the original map
     */
    public void setUserMetadata(Map<String, String> userMetadata) {
        if (userMetadata == null) {
            this.userMetadata = null;
        } else {
            this.userMetadata = new TreeMap<String, String>(String.CASE_INSENSITIVE_ORDER);
            this.userMetadata.putAll(userMetadata);
        }
    }

    public String getUserMetadata(String name) {
        return userMetadata.get(name);
    }

    public String getDecodedUserMetadata(String name) {
        final String value = getUserMetadata(name);
        if (value == null) {
            return null;
        }
        try {
            return URLDecoder.decode(value, "UTF-8");
        } catch (UnsupportedEncodingException e) {
            throw new RuntimeException("UTF-8 encoding isn't supported on this system", e);
        }
    }

    public ExsMetadata addUserMetadata(String name, String value) {
        userMetadata.put(name, value);
        return this;
    }

    public ExsMetadata addEncodedUserMetadata(String name, String value) {
        if (value == null) {
            return this;
        }

        try {
            final String encode = URLEncoder.encode(value, "UTF-8");
            return addUserMetadata(name, encode);
        } catch (UnsupportedEncodingException e) {
            throw new RuntimeException("UTF-8 encoding isn't supported on this system", e);
        }
    }

    public ExsMetadata withType(String type) {
        setType(type);
        return this;
    }

    public ExsMetadata withContentType(String contentType) {
        setContentType(contentType);
        return this;
    }

    public ExsMetadata withContentLength(int contentLength) {
        setContentLength((long) contentLength);
        return this;
    }

    public ExsMetadata withContentLength(long contentLength) {
        setContentLength(contentLength);
        return this;
    }

    public ExsMetadata withContentHash(String contentHash) {
        setContentHash(contentHash);
        return this;
    }

    public ExsMetadata withFrom(String from) {
        setFrom(from);
        return this;
    }

    public ExsMetadata withTo(String to) {
        setTo(to);
        return this;
    }

    public ExsMetadata withRepo(String repo) {
        setRepo(repo);
        return this;
    }

    public ExsMetadata withId(String id) {
        setId(id);
        return this;
    }

    public ExsMetadata withLastModified(Date date) {
        setLastModified(date);
        return this;
    }

    public ExsMetadata withLastModified(long date) {
        setLastModified(new Date(date));
        return this;
    }

    public Map<String, String> toHeaders() {
        Map<String, String> headers = new HashMap<>();
        // headers.put(HEADER_EXS_TYPE, getType())
        Optional.ofNullable(getType()).ifPresent(h -> headers.put(HEADER_EXS_TYPE, h));
        // headers.put(HEADER_EXS_TO, getTo())
        Optional.ofNullable(getTo()).ifPresent(h -> headers.put(HEADER_EXS_TO, h));
        Optional.ofNullable(getRepo()).ifPresent(h -> headers.put(HEADER_REPO, h));
        Optional.ofNullable(getContentType()).ifPresent(h -> headers.put(HEADER_CONTENT_TYPE, h));
        // headers.put(HEADER_CONTENT_TYPE, getContentType())
        Optional.ofNullable(getContentHash()).ifPresent(h -> headers.put(HEADER_NAME_CONTENT_HASH, h));
        Optional.ofNullable(getLastModified()).ifPresent(d -> headers.put(HEADER_LAST_MODIFIED, DateUtils.formatDate(d)));

        headers.putAll(getUmdHeaders(getUserMetadata()));
        return headers;
    }

    public void toHeaders(BiConsumer<String, String> headerSetter) {
        //headerSetter.accept(HEADER_EXS_TYPE, getType());
        Optional.ofNullable(getType()).ifPresent(h -> headerSetter.accept(HEADER_EXS_TYPE, h));
        //headerSetter.accept(HEADER_EXS_TO, getTo());
        Optional.ofNullable(getTo()).ifPresent(h -> headerSetter.accept(HEADER_EXS_TO, h));
        Optional.ofNullable(getRepo()).ifPresent(h -> headerSetter.accept(HEADER_REPO, h));
        Optional.ofNullable(getContentType()).ifPresent(h -> headerSetter.accept(HEADER_CONTENT_TYPE, h));
        // headerSetter.accept(HEADER_CONTENT_TYPE, getContentType())
        Optional.ofNullable(getContentHash()).ifPresent(h -> headerSetter.accept(HEADER_NAME_CONTENT_HASH, h));

        Optional.ofNullable(getLastModified()).ifPresent(d -> headerSetter.accept(HEADER_LAST_MODIFIED, DateUtils.formatDate(d)));

        toUmdHeaders(getUserMetadata(), headerSetter);
    }

    public static Map<String, String> getUmdHeaders(Map<String, String> userMetadata) {
        Map<String, String> headers = new HashMap<>();
        userMetadata.forEach((k, v) -> headers.put(HEADER_META_PREFIX + k, v));
        return headers;
    }

    public static void toUmdHeaders(Map<String, String> userMetadata, BiConsumer<String, String> headerSetter) {
        userMetadata.forEach((k, v) -> headerSetter.accept(HEADER_META_PREFIX + k, v));
    }

    public static ExsMetadata fromHeaders(Map<String, String> headers) {
        final ExsMetadata metadata = new ExsMetadata();
        Optional.ofNullable(headers.get(HEADER_EXS_ID)).ifPresent(metadata::setId);
        Optional.ofNullable(headers.get(HEADER_EXS_TYPE)).ifPresent(metadata::setType);
        Optional.ofNullable(headers.get(HEADER_EXS_FROM)).ifPresent(metadata::setFrom);
        Optional.ofNullable(headers.get(HEADER_EXS_TO)).ifPresent(metadata::setTo);
        Optional.ofNullable(headers.get(HEADER_REPO)).ifPresent(metadata::setRepo);
        Optional.ofNullable(headers.get(HEADER_CONTENT_TYPE)).ifPresent(metadata::setContentType);
        Optional.ofNullable(headers.get(HEADER_CONTENT_LENGTH)).map(Long::parseLong).ifPresent(metadata::setContentLength);
        Optional.ofNullable(headers.get(HEADER_NAME_CONTENT_HASH)).ifPresent(metadata::setContentHash);
        Optional.ofNullable(headers.get(HEADER_LAST_MODIFIED)).map(DateUtils::parseDate).ifPresent(metadata::setLastModified);

        final Map<String, String> userMetadata = getUserMetadata(headers);
        metadata.setUserMetadata(userMetadata);

        return metadata;
    }

    public static Map<String, String> getUserMetadata(Map<String, String> headers) {
        Map<String, String> userMetadata = new TreeMap<String, String>(String.CASE_INSENSITIVE_ORDER);
        headers.forEach((k, v) -> {
            String key = getUserMetadataKey(k);
            if (key != null) {
                userMetadata.put(key, v);
            }
        });
        return userMetadata;
    }

    protected static String getUserMetadataKey(String headerName) {
        if (headerName.toLowerCase().startsWith(HEADER_META_PREFIX)) {
            return headerName.substring(HEADER_META_PREFIX_LENGTH);
        }
        return null;
    }
}
