package org.opoo.ootp.client;

import lombok.extern.slf4j.Slf4j;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Consumer;

@Slf4j
public class SmartMessageClient implements MessageClient {
    public static final String META_FILE_REF = "file-ref";

    private final MessageClient messageClient;
    private final FileClient fileClient;

    public SmartMessageClient(MessageClient messageClient, FileClient fileClient) {
        this.messageClient = messageClient;
        this.fileClient = fileClient;
    }

    public SmartMessageClient(OotpClient ootpClient) {
        this.messageClient = ootpClient.getMessageClient();
        this.fileClient = ootpClient.getFileClient();
    }

    /**
     * 支持文件及大报文的发送，接收时必须单独接收，使用 {@link #getMessage(String)} 读取消息内容。
     *
     * @param message 消息
     * @return 消息ID
     * @throws OotpException
     */
    @Override
    public String send(ExsMessage message) throws OotpException {
        final ExsMetadata metadata = message.getMetadata();
        final Long contentLength = metadata.getContentLength();
        final String contentType = metadata.getContentType();
        final String to = metadata.getTo();
        final String type = metadata.getType();
        final Map<String, String> userMetadata = new HashMap<>(metadata.getUserMetadata());

        final boolean isText = Optional.ofNullable(contentType).map(String::toLowerCase)
                .filter(s -> s.startsWith("text/") || (s.startsWith("application/") && s.contains("json")))
                .isPresent();

        // 文本消息，并且小于 15000 时走普通消息，否则走文件
        if (isText && (contentLength == null || contentLength <= 15000)) {
            log.info("文本消息并且文件小于 15000，发送普通消息：{}, {}", contentType, contentLength);
            return messageClient.send(message);
        }

        log.info("不是文本消息或者消息体过大，使用文件传输：{}, {}", contentType, contentLength);
        processFileName(message);
        final String fileId;
        try {
            fileId = fileClient.upload(message);
        } catch (IOException e) {
            throw new OotpException("文件上传失败", e);
        }

        String text = META_FILE_REF + ":" + fileId;
        final ExsMetadata exsMetadata = new ExsMetadata()
                .withContentType("text/plain;charset=UTF-8")
                .withContentLength(text.getBytes(StandardCharsets.UTF_8).length)
                .withTo(to)
                .withType(type);
        // 将 meta 复制过来，便于查找问题
        exsMetadata.setUserMetadata(userMetadata);
        // 文件 ID 存取来
        exsMetadata.addUserMetadata(META_FILE_REF, fileId);
        // 明确表示不必加密
        exsMetadata.addUserMetadata(ExsCodec.META_CODEC_REQUIRED, "false");

        return messageClient.send(text, exsMetadata);
    }

    void processFileName(ExsMessage message) {
        final ExsMetadata metadata = message.getMetadata();
        if (metadata.getUserMetadata(Metadata.META_FILE_NAME) == null) {
            metadata.addUserMetadata(Metadata.META_FILE_NAME, "file-ref.bin")
                    .withLastModified(new Date())
            /*.addUserMetadata(Metadata.META_LAST_MODIFIED, DateUtils.formatDate(new Date()))*/;
        }
    }

    @Override
    public PollResult poll(PollRequest request) throws OotpException {
        return messageClient.poll(request);
    }

    @Override
    public ExsMessage getMessage(String id) throws OotpException {
        final ExsMessage message = messageClient.getMessage(id);
        final String fileRef = message.getMetadata().getUserMetadata(META_FILE_REF);
        if (fileRef == null) {
            return message;
        }
        try {
            final ExsMessage file = fileClient.getFile(fileRef);
            file.getMetadata().setId(id);
            return file;
        } catch (IOException e) {
            throw new OotpException("下载文件出错", e);
        }
    }

    @Override
    public ExsMessageInfo getMessageInfo(String id) throws OotpException {
        return messageClient.getMessageInfo(id);
    }

    @Override
    public int ack(List<String> ids) throws OotpException {
        return messageClient.ack(ids);
    }

    @Override
    public int ret(List<String> ids) throws OotpException {
        return messageClient.ret(ids);
    }

    @Override
    public int ret(List<String> ids, Consumer<Object> requestConfigurer) throws OotpException {
        return messageClient.ret(ids, requestConfigurer);
    }
}
