package org.opoo.ootp.client.httpclient;

import lombok.RequiredArgsConstructor;
import lombok.Setter;
import lombok.extern.slf4j.Slf4j;
import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.HttpRequestInterceptor;
import org.apache.http.RequestLine;
import org.apache.http.protocol.HttpContext;
import org.opoo.ootp.client.ExsCodec;
import org.opoo.ootp.client.ExsMetadata;
import org.opoo.ootp.client.Metadata;
import org.opoo.ootp.client.OotpClient;

import java.io.IOException;
import java.net.URI;

@Slf4j
@RequiredArgsConstructor
public class ExsCodecHttpRequestInterceptor implements HttpRequestInterceptor {
    private final ExsCodec exsCodec;
    @Setter
    private String pathPrefix = OotpClient.OPEN_API_CONTEXT_PATH;

    @Override
    public void process(HttpRequest httpRequest, HttpContext httpContext) throws HttpException, IOException {
        if (skip(httpRequest) || !(httpRequest instanceof HttpEntityEnclosingRequest)) {
            log.debug("无需加密处理: {}", httpRequest.getRequestLine().getUri());
            return;
        }

        final HttpEntityEnclosingRequest request = (HttpEntityEnclosingRequest) httpRequest;
        final ExsMetadata exsMetadata = new ExsMetadata();
        final Header firstHeader = request.getFirstHeader(Metadata.HEADER_EXS_TO);
        if (firstHeader != null) {
            exsMetadata.setTo(firstHeader.getValue());
        }

        final HttpEntity encode = exsCodec.encode(request.getEntity(), exsMetadata);
        request.setEntity(encode);

        exsMetadata.toHeaders(request::setHeader);
    }

    protected boolean skip(HttpRequest request) {
        final RequestLine requestLine = request.getRequestLine();
        if (!"POST".equals(requestLine.getMethod())) {
            return true;
        }
        if (pathPrefix == null) {
            return true;
        }

        final String uri = requestLine.getUri();
        final String path = URI.create(uri).getPath();
        return !path.startsWith(pathPrefix);
    }
}
