package org.opoo.ootp.client.impl;

import cn.hutool.core.io.IoUtil;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.Setter;
import org.apache.http.HttpEntity;
import org.apache.http.HttpStatus;
import org.apache.http.StatusLine;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.BufferedHttpEntity;
import org.apache.http.entity.InputStreamEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.util.EntityUtils;
import org.opoo.ootp.client.ApiClient;
import org.opoo.ootp.client.ByteArrayBody;
import org.opoo.ootp.client.EntityBody;
import org.opoo.ootp.client.ExsBody;
import org.opoo.ootp.client.ExsCodec;
import org.opoo.ootp.client.ExsMessage;
import org.opoo.ootp.client.ExsMetadata;
import org.opoo.ootp.client.OotpClient;
import org.opoo.ootp.client.OotpException;
import org.opoo.ootp.client.StringBody;

import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;


public class ApiClientImpl implements ApiClient {
    private final CloseableHttpClient httpClient;
    private final String baseUrl;
    private final ObjectMapper objectMapper;

    @Setter
    private ExsCodec codec;

    public ApiClientImpl(final URI endpoint, final CloseableHttpClient httpClient,
                         final ObjectMapper objectMapper, final String openApiContextPath) {
        this.httpClient = httpClient;
        this.objectMapper = objectMapper;
        this.baseUrl = endpoint.toString() + Optional.ofNullable(openApiContextPath).orElse(OotpClient.OPEN_API_CONTEXT_PATH);
    }

    @Override
    public ExsMessage post(String path, ExsMessage request) throws OotpException {
        final ExsMetadata metadata = Objects.requireNonNull(request.getMetadata(), "metadata is required.");
        Objects.requireNonNull(metadata.getContentType(), "contentType is required.");
        final ExsBody body = request.getBody();
        Objects.requireNonNull(body, "body is required.");

        final HttpPost post = new HttpPost(baseUrl + path);
        try {
            final HttpEntity httpEntity = AbstractMessageClient.toEntity(body, metadata);
            if (codec != null) {
                post.setEntity(codec.encode(httpEntity, metadata));
                metadata.toHeaders(post::setHeader);
            } else {
                post.setEntity(httpEntity);
            }
        } catch (IOException e) {
            throw new OotpException("处理请求输入时出错", e);
        }

        try (final CloseableHttpResponse response = httpClient.execute(post)) {
            final StatusLine statusLine = response.getStatusLine();
            final int statusCode = statusLine.getStatusCode();

            if (statusCode < HttpStatus.SC_OK || statusCode >= HttpStatus.SC_MULTIPLE_CHOICES) {
                final String string = EntityUtils.toString(response.getEntity(), StandardCharsets.UTF_8);
                throw new OotpException("文件下载失败：" + statusCode + " - " + statusLine.getReasonPhrase() + "\n"
                        + string, statusCode, statusLine.getReasonPhrase());
            }

            final HttpEntity entity;
            if (codec != null) {
                final Map<String, String> headers = new HashMap<>();
                Arrays.stream(response.getAllHeaders()).forEach(h -> headers.putIfAbsent(h.getName().toLowerCase(), h.getValue()));
                final ExsMetadata responseMetadata = ExsMetadata.fromHeaders(headers);
                entity = codec.decode(response.getEntity(), responseMetadata);
            } else {
                entity = response.getEntity();
            }

            final BufferedHttpEntity httpEntity = new BufferedHttpEntity(entity);
            return new ExsMessage(new EntityBody(httpEntity), metadata);
        } catch (IOException ex) {
            throw new OotpException("API调用失败：" + ex.getMessage(), ex);
        }
    }

    @Override
    public <T> T post(String path, Object request, Class<T> resultType) {
        Objects.requireNonNull(request, "request is required.");
        Objects.requireNonNull(resultType, "resultType is required.");
        Objects.requireNonNull(path, "path is required.");

        final ExsMessage exsMessage;
        if (request instanceof ExsMessage) {
            exsMessage = (ExsMessage) request;
        } else {
            exsMessage = new ExsMessage();
            setMessageBody(exsMessage, request);
            exsMessage.setMetadata(new ExsMetadata());
            exsMessage.getMetadata().setContentType("application/json; charset=utf-8");
        }

        final ExsMessage result = post(path, exsMessage);
        return convertResult(result, resultType);
    }

    private void setMessageBody(ExsMessage exsMessage, Object request) {
        if (request instanceof ExsBody) {
            exsMessage.setBody((ExsBody) request);
        } else if (request instanceof String) {
            exsMessage.setBody(new StringBody((String) request, StandardCharsets.UTF_8));
        } else if (request instanceof byte[]) {
            exsMessage.setBody(new ByteArrayBody((byte[]) request));
        } else if (request instanceof InputStream) {
            exsMessage.setBody(new EntityBody(new InputStreamEntity((InputStream) request)));
        } else {
            final byte[] bytes;
            try {
                bytes = objectMapper.writeValueAsBytes(request);
            } catch (JsonProcessingException e) {
                throw new OotpException("请求对象转换失败", e);
            }
            exsMessage.setBody(new ByteArrayBody(bytes));
        }
    }

    @SuppressWarnings("unchecked")
    private <T> T convertResult(ExsMessage result, Class<T> resultType) {
        try {
            if (ExsMessage.class.isAssignableFrom(resultType)) {
                return (T) result;
            } else if (ExsBody.class.isAssignableFrom(resultType)) {
                return (T) result.getBody();
            } else if (InputStream.class.isAssignableFrom(resultType)) {
                return (T) result.getBody().getContent();
            } else if (String.class.isAssignableFrom(resultType)) {
                return (T) IoUtil.read(result.getBody().getContent(), StandardCharsets.UTF_8);
            } else if (byte[].class.isAssignableFrom(resultType)) {
                return (T) IoUtil.readBytes(result.getBody().getContent());
            } else {
                return objectMapper.readValue(result.getBody().getContent(), resultType);
            }
        } catch (IOException e) {
            throw new OotpException("响应结果转换失败", e);
        }
    }
}
