package org.opoo.ootp.client.impl;

import lombok.extern.slf4j.Slf4j;

import java.io.FilterInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;

@Slf4j
class FileCachedInputStream extends FilterInputStream {
    private final Path file;

    /**
     * Creates a <code>FilterInputStream</code>
     * by assigning the  argument <code>in</code>
     * to the field <code>this.in</code> so as
     * to remember it for later use.
     *
     * @param in   the underlying input stream, or <code>null</code> if
     *             this instance is to be created without an underlying stream.
     * @param file cached file.
     */
    private FileCachedInputStream(InputStream in, Path file) {
        super(in);
        this.file = file;
    }

    FileCachedInputStream(Path file) throws IOException {
        super(Files.newInputStream(file));
        this.file = file;
    }

    @Override
    public void close() throws IOException {
        super.close();
        final boolean result = Files.deleteIfExists(file);
        log.debug("关闭时删除临时文件：{} - {}", file, result);
    }

    public Path getFile() {
        return file;
    }

    static FileCachedInputStream create(InputStream originalInputStream) throws IOException {
        return create(originalInputStream, "tmp.bin");
    }

    static FileCachedInputStream create(InputStream originalInputStream, String fileId) throws IOException {
        final Path path = Files.createTempFile("ootp", fileId);
        log.debug("复制到临时文件：{}", path);
        try (InputStream inputStream = originalInputStream) {
            Files.copy(inputStream, path, StandardCopyOption.REPLACE_EXISTING);
        }
        return new FileCachedInputStream(Files.newInputStream(path), path);
    }
}
