package org.opoo.ootp.client.spring;

import cn.hutool.core.io.IoUtil;
import lombok.RequiredArgsConstructor;
import lombok.Setter;
import org.opoo.ootp.client.ExsCodec;
import org.opoo.ootp.client.ExsMetadata;
import org.opoo.ootp.client.Metadata;
import org.opoo.ootp.client.OotpClient;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.HttpRequest;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.client.ClientHttpRequestExecution;
import org.springframework.http.client.ClientHttpRequestInterceptor;
import org.springframework.http.client.ClientHttpResponse;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.AbstractMap;
import java.util.Map;
import java.util.stream.Collectors;

@RequiredArgsConstructor
public class ExsCodecClientHttpRequestInterceptor implements ClientHttpRequestInterceptor {
    private final ExsCodec exsCodec;
    @Setter
    private String openApiContextPath;

    @Override
    public ClientHttpResponse intercept(HttpRequest request, byte[] body, ClientHttpRequestExecution execution) throws IOException {
        if (skip(request)) {
            return execution.execute(request, body);
        }

        final HttpHeaders headers = request.getHeaders();
        final MediaType contentType = headers.getContentType();

        final ExsMetadata exsMetadata = new ExsMetadata();
        final String extTo = headers.getFirst(Metadata.HEADER_EXS_TO);
        if (extTo != null) {
            exsMetadata.setTo(extTo);
        }

        final byte[] newBody;
        try (final InputStream inputStream = new ByteArrayInputStream(body);
             final InputStream encodedStream = exsCodec.encode(inputStream, exsMetadata)) {
            newBody = IoUtil.readBytes(encodedStream, false);
        }

        exsMetadata.toHeaders(headers::set);
        headers.setContentLength(newBody.length);
        headers.setContentType(contentType);

        final ClientHttpResponse response = execution.execute(request, newBody);

        final HttpHeaders responseHeaders = new HttpHeaders(response.getHeaders());
        final Map<String, String> map = responseHeaders.toSingleValueMap().entrySet().stream()
                .map(en -> new AbstractMap.SimpleEntry<>(en.getKey().toLowerCase(), en.getValue()))
                .collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue));
        final ExsMetadata responseExsMetadata = ExsMetadata.fromHeaders(map);
        final byte[] newResponseBody;
        try (final InputStream inputStream = exsCodec.decode(response.getBody(), responseExsMetadata)) {
            newResponseBody = IoUtil.readBytes(inputStream, false);
        }

        responseHeaders.setContentLength(newResponseBody.length);
        return new ClientHttpResponseWrapper(response, responseHeaders, newResponseBody);
    }

    protected boolean skip(HttpRequest request) {
        if (request.getMethod() != HttpMethod.POST) {
            return true;
        }
        final String path = request.getURI().getPath();
        final String pathPrefix = openApiContextPath != null ? openApiContextPath : OotpClient.OPEN_API_CONTEXT_PATH;
        return !path.startsWith(pathPrefix);
    }

    @RequiredArgsConstructor
    static class ClientHttpResponseWrapper implements ClientHttpResponse {
        private final ClientHttpResponse delegate;
        private final HttpHeaders headers;
        private final byte[] body;

        @Override
        public HttpStatus getStatusCode() throws IOException {
            return delegate.getStatusCode();
        }

        @Override
        public int getRawStatusCode() throws IOException {
            return delegate.getRawStatusCode();
        }

        @Override
        public String getStatusText() throws IOException {
            return delegate.getStatusText();
        }

        @Override
        public void close() {
            delegate.close();
        }

        @Override
        public InputStream getBody() throws IOException {
            return new ByteArrayInputStream(body);
        }

        @Override
        public HttpHeaders getHeaders() {
            return headers;
        }
    }
}
