package org.opoo.ootp.client.messaging;

import lombok.extern.slf4j.Slf4j;
import org.opoo.ootp.client.ExsClient;
import org.opoo.ootp.client.ExsMessage;
import org.springframework.messaging.Message;
import org.springframework.messaging.MessagingException;
import org.springframework.messaging.SubscribableChannel;
import org.springframework.messaging.converter.MessageConverter;

import java.util.List;

@Slf4j
public class DefaultExsMessageDispatcher implements PollableExsMessageDispatcher {
    private final SubscribableChannel subscribableChannel;
    private final MessageConverter messageConverter;
    private final ExsClient exsClient;

    private boolean defaultAutoAck = true;
    private boolean defaultFull = true;
    private int defaultLimit = 10;
    private boolean ignoreFailures = true;

    public DefaultExsMessageDispatcher(SubscribableChannel subscribableChannel, MessageConverter messageConverter, ExsClient exsClient) {
        this.subscribableChannel = subscribableChannel;
        this.messageConverter = messageConverter;
        this.exsClient = exsClient;
    }

    public boolean isDefaultAutoAck() {
        return defaultAutoAck;
    }

    public void setDefaultAutoAck(boolean defaultAutoAck) {
        this.defaultAutoAck = defaultAutoAck;
    }

    public boolean isDefaultFull() {
        return defaultFull;
    }

    public void setDefaultFull(boolean defaultFull) {
        this.defaultFull = defaultFull;
    }

    public int getDefaultLimit() {
        return defaultLimit;
    }

    public void setDefaultLimit(int defaultLimit) {
        this.defaultLimit = defaultLimit;
    }

    public boolean isIgnoreFailures() {
        return ignoreFailures;
    }

    public void setIgnoreFailures(boolean ignoreFailures) {
        this.ignoreFailures = ignoreFailures;
    }

    @Override
    public void dispatch(ExsMessage exsMessage) {
        final Message<?> toMessage = messageConverter.toMessage(exsMessage, null);

        if (toMessage == null) {
            if (isIgnoreFailures()) {
                log.warn("消息转换失败，转换后为空：{}", exsMessage);
                return;
            } else {
                throw new IllegalStateException("消息转换后为空：" + exsMessage);
            }
        }

        try {
            subscribableChannel.send(toMessage);
        } catch (RuntimeException e) {
            if (!this.isIgnoreFailures()) {
                if (e instanceof MessagingException && ((MessagingException) e).getFailedMessage() == null) {
                    throw new MessagingException(toMessage, "Failed to handle message", e);
                }
                throw e;
            } else {
                log.warn("消息分发失败。Suppressing Exception since 'ignoreFailures' is set to TRUE.", e);
            }
        }
    }

    @Override
    public void dispatch(List<ExsMessage> messages) {
        for (ExsMessage message: messages) {
            dispatch(message);
        }
    }

    @Override
    public void pollAndDispatch(int limit, boolean autoAck, boolean full) {
        final List<ExsMessage> messages = exsClient.poll(limit, autoAck, full);
        if (messages == null) {
            log.warn("轮询结果为 null");
            return;
        }

        final int size = messages.size();
        log.debug("轮询消息 {} 条", size);

        if (size == 0) {
            return;
        }

        dispatch(messages);

        if (size >= limit) {
            // 继续轮询，一次性全部处理完
            pollAndDispatch(limit, autoAck, full);
        }
    }

    @Override
    public void pollAndDispatch() {
        pollAndDispatch(defaultLimit, defaultAutoAck, defaultFull);
    }
}
