package org.opoo.ootp.client.messaging;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.opoo.ootp.client.ExsClient;
import org.opoo.ootp.client.ExsMessage;
import org.opoo.ootp.client.ExsModel;
import org.opoo.ootp.client.OotpException;
import org.springframework.messaging.Message;
import org.springframework.messaging.MessageHeaders;
import org.springframework.messaging.PollableChannel;
import org.springframework.messaging.converter.MessageConverter;

import java.util.List;
import java.util.Objects;
import java.util.Optional;

import static org.opoo.ootp.client.messaging.ExsMessageHeaders.DEFAULT_CONTENT_TYPE;
import static org.opoo.ootp.client.messaging.ExsMessageHeaders.TO;
import static org.opoo.ootp.client.messaging.ExsMessageHeaders.TYPE;
import static org.springframework.messaging.MessageHeaders.CONTENT_TYPE;


public class ExsMessageChannel implements PollableChannel {
    private final ExsClient exsClient;
    private final MessageConverter messageConverter;
    private final ObjectMapper objectMapper;

    public ExsMessageChannel(ExsClient exsClient) {
        this.exsClient = exsClient;

        final ExsMessageConverter messageConverter = new ExsMessageConverter();
        this.messageConverter = messageConverter;
        this.objectMapper = messageConverter.getObjectMapper();
    }

    public ExsMessageChannel(ExsClient exsClient, ExsMessageConverter messageConverter) {
        this.exsClient = exsClient;
        this.messageConverter = messageConverter;
        this.objectMapper = messageConverter.getObjectMapper();
    }

    public ExsMessageChannel(ExsClient exsClient, MessageConverter messageConverter, ObjectMapper objectMapper) {
        this.exsClient = exsClient;
        this.messageConverter = messageConverter;
        this.objectMapper = objectMapper;
    }

    @Override
    public boolean send(Message<?> message, long timeout) {
        final Object payload = message.getPayload();
        final MessageHeaders headers = message.getHeaders();
        final String to = headers.get(TO, String.class);
        final String contentType = (String) headers.getOrDefault(CONTENT_TYPE, DEFAULT_CONTENT_TYPE);
        final String type = Optional.ofNullable(headers.get(TYPE, String.class))
                .orElseGet(() ->
                    Optional.of(payload)
                        .filter(o -> o instanceof ExsModel)
                        .map(o -> ((ExsModel) o).getPrccod())
                        .orElse(null)
                );

        Objects.requireNonNull(type, "Header 'type' is required.");
        Objects.requireNonNull(to, "Header 'to' is required.");

        if (payload instanceof String) {
            exsClient.send(type, to, contentType, (String) payload);
        } else if (payload instanceof byte[]) {
            exsClient.send(type, to, contentType, (byte[]) payload);
        } else {
            // convert
            try {
                final byte[] bytes = objectMapper.writeValueAsBytes(payload);
                exsClient.send(type, to, contentType, bytes);
            } catch (JsonProcessingException e) {
                throw new OotpException("消息序列化失败", e);
            }
        }
        return true;
    }

    @Override
    public Message<?> receive() {
        return receive(INDEFINITE_TIMEOUT);
    }

    @Override
    public Message<?> receive(long timeout) {
        final List<ExsMessage> messages = exsClient.poll(1, true, true);

        if (messages != null && !messages.isEmpty()) {
            final ExsMessage message = messages.iterator().next();
            return messageConverter.toMessage(message, new ExsMessageHeaders());
        }
        return null;
    }

}
