package org.opoo.ootp.client.messaging;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.extern.slf4j.Slf4j;
import org.opoo.ootp.client.ExsMessage;
import org.opoo.ootp.client.ExsModel;
import org.springframework.messaging.Message;
import org.springframework.messaging.MessageHeaders;
import org.springframework.messaging.converter.MappingJackson2MessageConverter;
import org.springframework.messaging.converter.MessageConversionException;
import org.springframework.messaging.converter.MessageConverter;
import org.springframework.messaging.converter.SmartMessageConverter;
import org.springframework.messaging.support.MessageBuilder;
import org.springframework.util.StringUtils;

import java.nio.charset.StandardCharsets;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Objects;

import static org.opoo.ootp.client.messaging.ExsMessageHeaders.DEFAULT_CONTENT_TYPE;
import static org.opoo.ootp.client.messaging.ExsMessageHeaders.FROM;
import static org.opoo.ootp.client.messaging.ExsMessageHeaders.ID;
import static org.opoo.ootp.client.messaging.ExsMessageHeaders.ORIGINAL_MESSAGE;
import static org.opoo.ootp.client.messaging.ExsMessageHeaders.TYPE;
import static org.springframework.messaging.MessageHeaders.CONTENT_TYPE;

@Slf4j
public class ExsMessageConverter implements MessageConverter, SmartMessageConverter {
    private final MappingJackson2MessageConverter delegate;

    public ExsMessageConverter() {
        this.delegate = new MappingJackson2MessageConverter();
    }

    public void setObjectMapper(ObjectMapper objectMapper) {
        delegate.setObjectMapper(objectMapper);
    }

    public ObjectMapper getObjectMapper() {
        return delegate.getObjectMapper();
    }

    @Override
    public Object fromMessage(Message<?> message, Class<?> targetClass, Object conversionHint) {
        final Object payload = message.getPayload();
        final MessageHeaders headers = message.getHeaders();
        final boolean isStringPayload = (payload instanceof String);
        final boolean isByteArrayPayload = !isStringPayload && (payload instanceof byte[]);

        if (ExsMessage.class.isAssignableFrom(targetClass)) {
            // 如果有原始消息，则直接返回
            ExsMessage exsMessage = headers.get(ORIGINAL_MESSAGE, ExsMessage.class);

            if (exsMessage == null) {
                final String from = Objects.requireNonNull(headers.get(FROM, String.class), "Header 'from' is required.");
                final String id = Objects.requireNonNull(headers.get(ID, String.class), "Header 'id' is required.");
                final String type = Objects.requireNonNull(headers.get(TYPE, String.class), "Header 'type' is required.");
                final String contentType = headers.get(CONTENT_TYPE, String.class);

                // 否则构建 ExsMessage
                exsMessage = new ExsMessage();
                exsMessage.setFrom(from);
                exsMessage.setId(id);
                exsMessage.setType(type);
                exsMessage.setContentType(contentType);

                if (isStringPayload) {
                    exsMessage.setBody((String)payload);
                } else if (isByteArrayPayload) {
                    exsMessage.setBody(new String((byte[]) payload, StandardCharsets.UTF_8));
                } else {
                    try {
                        exsMessage.setBody(getObjectMapper().writeValueAsString(payload));
                    } catch (JsonProcessingException e) {
                        throw new MessageConversionException("参数转换失败", e);
                    }
                }
            }

            return exsMessage;
        }

        if (message.getHeaders().get(CONTENT_TYPE) == null) {
            log.debug("Add 'contentType' header: {}", message);
            message = MessageBuilder.fromMessage(message).setHeader(CONTENT_TYPE, DEFAULT_CONTENT_TYPE).build();
        }

        return delegate.fromMessage(message, targetClass, conversionHint);
    }

    @Override
    public Message<?> toMessage(Object payload, MessageHeaders headers, Object conversionHint) {
        if (payload instanceof ExsMessage) {
            log.debug("Converting ExsMessage to Message: {}", payload);
            ExsMessage exsMessage = (ExsMessage) payload;
            return MessageBuilder.withPayload(exsMessage.getBody())
                    .copyHeaders(headers)
                    .setHeader(TYPE, exsMessage.getType())
                    .setHeader(FROM, exsMessage.getFrom())
                    .setHeader(CONTENT_TYPE, exsMessage.getContentType())
                    .setHeader(ID, exsMessage.getId())
                    .setHeader(ORIGINAL_MESSAGE, exsMessage)
                    .setHeaderIfAbsent(CONTENT_TYPE, DEFAULT_CONTENT_TYPE)
                    .build();
        }

        if (payload instanceof ExsModel) {
            final String prccod = ((ExsModel) payload).getPrccod();
            if (StringUtils.hasText(prccod)) {
                headers = addHeaderIfAbsent(headers, TYPE, prccod);
            }
        }

        headers = addHeaderIfAbsent(headers, CONTENT_TYPE, DEFAULT_CONTENT_TYPE);
        return delegate.toMessage(payload, headers, conversionHint);
    }

    @Override
    public Object fromMessage(Message<?> message, Class<?> targetClass) {
        return fromMessage(message, targetClass, null);
    }

    @Override
    public Message<?> toMessage(Object payload, MessageHeaders headers) {
        return toMessage(payload, headers, null);
    }

    protected MessageHeaders addHeaderIfAbsent(MessageHeaders headers, String name, String value) {
        if (headers == null) {
            log.debug("Create new MessageHeaders and add header '{}': {}", name, value);
            return new ExsMessageHeaders(Collections.singletonMap(name, value));
        }

        if (headers.get(name) == null) {
            log.debug("Update MessageHeaders add header '{}': {}", name, value);
            Map<String, Object> map = new LinkedHashMap<>(headers);
            map.put(name, value);
            return new ExsMessageHeaders(map);
        }

        return headers;
    }
}
