package org.opoo.ootp.client.messaging;

import org.springframework.messaging.Message;
import org.springframework.messaging.MessageChannel;
import org.springframework.messaging.MessageHeaders;
import org.springframework.messaging.MessagingException;
import org.springframework.messaging.core.AbstractMessageSendingTemplate;
import org.springframework.messaging.support.MessageBuilder;
import org.springframework.util.Assert;

public class ExsMessageTemplate extends AbstractMessageSendingTemplate<MessageChannel> {
    private String defaultTo;

    /**
     * 设置消息默认的接收方，当前系统只向一个接收方发送消息时，可以配置此项，简化消息发送方法。
     * 消息在发送时未指定接收方时，取当前默认接收方的值。
     * @param to 默认的接收方。
     */
    public void setDefaultTo(String to) {
        this.defaultTo = to;
    }

    /**
     * 获取默认的的接收方。如果没有配置默认接收方，则报错。
     * @return 默认接收方
     */
    public String getRequiredDefaultTo() {
        Assert.state(this.defaultTo != null, "No 'defaultTo' configured");
        return this.defaultTo;
    }

    @Override
    protected void doSend(MessageChannel destination, Message<?> message) {
        final MessageHeaders headers = message.getHeaders();
        if (headers.get(ExsMessageHeaders.TO, String.class) == null) {
            message = MessageBuilder.fromMessage(message).setHeader(ExsMessageHeaders.TO, getRequiredDefaultTo()).build();
        }
        destination.send(message);
    }

    /**
     * 转换并发送消息。底层仍然调用 {@link org.opoo.ootp.client.MessageClient} 来发送消息。
     * @param type 消息业务类型
     * @param to 接收方
     * @param payload 消息体，任意类型，通常对象会转成 JSON 报文
     * @throws MessagingException 消息发送异常
     */
    public void convertAndSend(String type, String to, Object payload) throws MessagingException {
        final Message<Object> message = MessageBuilder.withPayload(payload)
                .setHeader(ExsMessageHeaders.TYPE, type)
                .setHeader(ExsMessageHeaders.TO, to)
                .setHeader(ExsMessageHeaders.CONTENT_TYPE, "application/json")
                .build();
        send(message);
    }

    /**
     * 转换并发送消息。底层仍然调用 {@link org.opoo.ootp.client.MessageClient} 来发送消息。使用默认接收方。
     * @param type 消息业务类型
     * @param payload 消息体
     */
    public void convertAndSend(String type, Object payload) {
        convertAndSend(type, null, payload);
    }
}
