package org.opoo.ootp.client.messaging;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.extern.slf4j.Slf4j;
import org.opoo.ootp.client.ByteArrayBody;
import org.opoo.ootp.client.ExsBody;
import org.opoo.ootp.client.ExsMessage;
import org.opoo.ootp.client.ExsMetadata;
import org.opoo.ootp.client.ExsModel;
import org.opoo.ootp.client.StringBody;
import org.springframework.messaging.Message;
import org.springframework.messaging.MessageHeaders;
import org.springframework.messaging.converter.MappingJackson2MessageConverter;
import org.springframework.messaging.converter.MessageConversionException;
import org.springframework.messaging.converter.MessageConverter;
import org.springframework.messaging.converter.SmartMessageConverter;
import org.springframework.messaging.support.MessageBuilder;
import org.springframework.messaging.support.MessageHeaderAccessor;
import org.springframework.util.MimeType;
import org.springframework.util.MimeTypeUtils;
import org.springframework.util.StreamUtils;
import org.springframework.util.StringUtils;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import static org.opoo.ootp.client.messaging.ExsMessageHeaders.CONTENT_HASH;
import static org.opoo.ootp.client.messaging.ExsMessageHeaders.CONTENT_LENGTH;
import static org.opoo.ootp.client.messaging.ExsMessageHeaders.DEFAULT_CONTENT_TYPE;
import static org.opoo.ootp.client.messaging.ExsMessageHeaders.FROM;
import static org.opoo.ootp.client.messaging.ExsMessageHeaders.ID;
import static org.opoo.ootp.client.messaging.ExsMessageHeaders.ORIGINAL_MESSAGE;
import static org.opoo.ootp.client.messaging.ExsMessageHeaders.TO;
import static org.opoo.ootp.client.messaging.ExsMessageHeaders.TYPE;
import static org.springframework.messaging.MessageHeaders.CONTENT_TYPE;

@Slf4j
public class ExsMessageConverter implements MessageConverter, SmartMessageConverter {
    private final MappingJackson2MessageConverter delegate;

    public ExsMessageConverter() {
        this.delegate = new MappingJackson2MessageConverter();
    }

    public void setObjectMapper(ObjectMapper objectMapper) {
        delegate.setObjectMapper(objectMapper);
    }

    public ObjectMapper getObjectMapper() {
        return delegate.getObjectMapper();
    }

    @Override
    public Object fromMessage(Message<?> message, Class<?> targetClass, Object conversionHint) {
        final Object payload = message.getPayload();
        final MessageHeaders headers = message.getHeaders();
        final boolean isStringPayload = (payload instanceof String);
        final boolean isByteArrayPayload = !isStringPayload && (payload instanceof byte[]);

        if (ExsMessage.class.isAssignableFrom(targetClass)) {
            // 如果有原始消息，则直接返回
            ExsMessage exsMessage = headers.get(ORIGINAL_MESSAGE, ExsMessage.class);

            if (exsMessage == null) {
                final String from = /*Objects.requireNonNull(*/headers.get(FROM, String.class)/*, "Header 'from' is required.")*/;
                final String id = /*Objects.requireNonNull(*/headers.get(ID, String.class)/*, "Header 'id' is required.")*/;
                //final String type = Objects.requireNonNull(headers.get(TYPE, String.class), "Header 'type' is required.");
                //final String contentType = (String) headers.getOrDefault(CONTENT_TYPE, DEFAULT_CONTENT_TYPE);
                final String contentType = headers.get(CONTENT_TYPE, String.class);
                final String to = headers.get(TO, String.class);
                final String contentHash = headers.get(CONTENT_HASH, String.class);
                final Number contentLength = headers.get(CONTENT_LENGTH, Number.class);
                final String type = Optional.ofNullable(headers.get(TYPE, String.class))
                        .orElseGet(() ->
                                Optional.of(payload)
                                        .filter(o -> o instanceof ExsModel)
                                        .map(o -> ((ExsModel) o).getPrccod())
                                        .orElse(null)
                        );
                // 无论发出还是接收的消息，其 type 都是必须的
                Objects.requireNonNull(type, "Header 'type' is required.");

                // 否则构建 ExsMessage
                final ExsMetadata exsMetadata = new ExsMetadata();
                exsMetadata.setFrom(from);
                exsMetadata.setId(id);
                exsMetadata.setType(type);
                exsMetadata.setTo(to);
                exsMetadata.setContentHash(contentHash);
                exsMetadata.setContentType(contentType);
                exsMetadata.setContentLength(contentLength != null ? contentLength.longValue() : null);

                headers.keySet().stream().filter(s -> s.toLowerCase().startsWith("meta-"))
                        .forEach(s -> exsMetadata.addUserMetadata(s, headers.get(s, String.class)));

                ExsBody body;
                if (isStringPayload) {
                    body = new StringBody((String) payload);
                } else if (isByteArrayPayload) {
                    body = new ByteArrayBody((byte[]) payload);
                } else {
                    body = new JsonBody(getObjectMapper(), payload);
                }

                exsMessage = new ExsMessage(body, exsMetadata);
            }

            return exsMessage;
        }

        if (message.getHeaders().get(CONTENT_TYPE) == null) {
            log.debug("Add 'contentType' header: {}", message);
            message = MessageBuilder.fromMessage(message).setHeader(CONTENT_TYPE, DEFAULT_CONTENT_TYPE).build();
        }

        // 从 Content-Type 中获取字符集
        // 与发送功能保持一致，默认使用 UTF-8 字符集
        final Charset charset = Optional.ofNullable(MessageHeaderAccessor.getAccessor(message))
                .map(MessageHeaderAccessor::getContentType)
                .map(MimeType::getCharset)
                .orElseGet(() ->
                        //如果不能通过 MessageHeaderAccessor 获取 Content-Type 则直接获取
                        Optional.ofNullable(headers.get(CONTENT_TYPE, String.class))
                                .map(MimeTypeUtils::parseMimeType)
                                .map(MimeType::getCharset)
                                .orElse(StandardCharsets.UTF_8)
                );

        log.debug("ContentType charset: {}", charset);

        if (byte[].class.equals(targetClass)) {
            if (isByteArrayPayload) {
                return payload;
            } else if (isStringPayload) {
                return ((String)payload).getBytes(charset);
            } else {
                try {
                    return getObjectMapper().writeValueAsBytes(payload);
                } catch (JsonProcessingException e) {
                    throw new MessageConversionException(message, "消息转换失败", e);
                }
            }
        }

        if (String.class.equals(targetClass)) {
            if (isStringPayload) {
                return payload;
            } else if (isByteArrayPayload) {
                return new String((byte[]) payload, charset);
            } else {
                try {
                    return getObjectMapper().writeValueAsString(payload);
                } catch (JsonProcessingException e) {
                    throw new MessageConversionException(message, "消息转换失败", e);
                }
            }
        }

        return delegate.fromMessage(message, targetClass, conversionHint);
    }

    @Override
    public Message<?> toMessage(Object payload, MessageHeaders headers, Object conversionHint) {
        if (payload instanceof ExsMessage) {
            log.debug("Converting ExsMessage to Message: {}", payload);

            final ExsMessage exsMessage = (ExsMessage) payload;
            final ExsMetadata metadata = exsMessage.getMetadata();
            final ExsBody body = exsMessage.getBody();

            final Object newPayload;
            if (body instanceof StringBody) {
                newPayload = ((StringBody) body).getString();
            } else {
                try (final InputStream inputStream = body.getContent()) {
                    newPayload = StreamUtils.copyToByteArray(inputStream);
                } catch (IOException e) {
                    throw new MessageConversionException("消息无法转换", e);
                }
            }

            final MessageBuilder<Object> messageBuilder = MessageBuilder.withPayload(newPayload)
                    .copyHeaders(headers)
                    .setHeader(TYPE, metadata.getType())
                    .setHeader(FROM, metadata.getFrom())
                    .setHeader(TO, metadata.getTo())
                    .setHeader(CONTENT_TYPE, metadata.getContentType())
                    //.setHeader(CONTENT_HASH, metadata.getContentHash())
                    //.setHeader(CONTENT_LENGTH, metadata.getContentLength())
                    .setHeader(ID, metadata.getId())
                    .setHeader(ORIGINAL_MESSAGE, exsMessage)
                    .setHeaderIfAbsent(CONTENT_TYPE, DEFAULT_CONTENT_TYPE);

            Optional.ofNullable(metadata.getContentHash()).ifPresent(s -> messageBuilder.setHeader(CONTENT_HASH, s));
            Optional.ofNullable(metadata.getContentLength()).ifPresent(v -> messageBuilder.setHeader(CONTENT_LENGTH, v));
            metadata.getUserMetadata().forEach((k,v) -> messageBuilder.setHeader("meta-" + k, v));

            return messageBuilder.build();
        }

        if (payload instanceof ExsModel) {
            final String prccod = ((ExsModel) payload).getPrccod();
            if (StringUtils.hasText(prccod)) {
                headers = addHeaderIfAbsent(headers, TYPE, prccod);
            }
        }

        headers = addHeaderIfAbsent(headers, CONTENT_TYPE, DEFAULT_CONTENT_TYPE);
        return delegate.toMessage(payload, headers, conversionHint);
    }

    @Override
    public Object fromMessage(Message<?> message, Class<?> targetClass) {
        return fromMessage(message, targetClass, null);
    }

    @Override
    public Message<?> toMessage(Object payload, MessageHeaders headers) {
        return toMessage(payload, headers, null);
    }

    protected MessageHeaders addHeaderIfAbsent(MessageHeaders headers, String name, String value) {
        if (headers == null) {
            log.debug("Create new MessageHeaders and add header '{}': {}", name, value);
            return new ExsMessageHeaders(Collections.singletonMap(name, value));
        }

        if (headers.get(name) == null) {
            log.debug("Update MessageHeaders add header '{}': {}", name, value);
            Map<String, Object> map = new LinkedHashMap<>(headers);
            map.put(name, value);
            return new ExsMessageHeaders(map);
        }

        return headers;
    }

    protected static class JsonBody implements ExsBody {
        private final ObjectMapper objectMapper;
        private final Object payload;

        public JsonBody(ObjectMapper objectMapper, Object payload) {
            this.objectMapper = objectMapper;
            this.payload = payload;
        }

        @Override
        public InputStream getContent() throws IOException, UnsupportedOperationException {
            final byte[] bytes = objectMapper.writeValueAsBytes(payload);
            return new ByteArrayInputStream(bytes);
        }

        @Override
        public void writeTo(OutputStream outputStream) throws IOException {
            objectMapper.writeValue(outputStream, payload);
        }
    }
}
