package org.opoo.ootp.client.messaging;

import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.scheduling.TaskScheduler;
import org.springframework.scheduling.concurrent.ThreadPoolTaskScheduler;
import org.springframework.util.Assert;

import java.time.Duration;
import java.util.concurrent.TimeUnit;

/**
 * 使用线程池定时轮询消息并分发。
 */
@Slf4j
public class ExsMessageListenerContainer implements InitializingBean {
    private final PollableExsMessageDispatcher messageDispatcher;
    private final TaskScheduler taskScheduler;
    private Duration delay = Duration.ofMinutes(5L);
    private Duration initialDelay = Duration.ofMinutes(1L);
    private boolean autoStart = true;

    private boolean running = false;

    public ExsMessageListenerContainer(PollableExsMessageDispatcher messageDispatcher, TaskScheduler taskScheduler) {
        this.messageDispatcher = messageDispatcher;
        this.taskScheduler = taskScheduler;
    }

    public Duration getDelay() {
        return delay;
    }

    public void setDelay(Duration delay) {
        this.delay = delay;
    }

    public Duration getInitialDelay() {
        return initialDelay;
    }

    public void setInitialDelay(Duration initialDelay) {
        this.initialDelay = initialDelay;
    }

    public boolean isAutoStart() {
        return autoStart;
    }

    public void setAutoStart(boolean autoStart) {
        this.autoStart = autoStart;
    }

    public PollableExsMessageDispatcher getMessageDispatcher() {
        return messageDispatcher;
    }

    public TaskScheduler getTaskScheduler() {
        return taskScheduler;
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        Assert.notNull(messageDispatcher, "messageDispatcher is required.");
        Assert.notNull(taskScheduler, "taskScheduler is required.");

        if (autoStart) {
            start();
        }
    }

    public void start() {
        if (!running) {
            log.debug("启动消息监听...");
            Runnable task = messageDispatcher::pollAndDispatch;

            if (taskScheduler instanceof ThreadPoolTaskScheduler) {
                ((ThreadPoolTaskScheduler) taskScheduler).getScheduledThreadPoolExecutor()
                        .scheduleWithFixedDelay(task, initialDelay.toMillis(), delay.toMillis(), TimeUnit.MILLISECONDS);
            } else {
                taskScheduler.scheduleWithFixedDelay(task, delay);
            }
            running = true;
        } else {
            log.warn("消息监听程序已经启动");
        }
    }

}
