package org.opoo.ootp.client.messaging;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.messaging.Message;
import org.springframework.messaging.MessageHandler;
import org.springframework.messaging.MessageHeaders;
import org.springframework.messaging.MessagingException;
import org.springframework.util.Assert;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;

/**
 * 根据类型处理的消息处理器。
 */
public abstract class AbstractExsMessageHandler implements MessageHandler {
    protected final Logger log = LoggerFactory.getLogger(getClass());
    private final List<String> supportedTypes;

    public AbstractExsMessageHandler(String supportedType) {
        Assert.notNull(supportedType, "支持的消息类型不能为空");
        this.supportedTypes = Collections.singletonList(supportedType);
    }

    public AbstractExsMessageHandler(String... supportedTypes) {
        Assert.notEmpty(supportedTypes, "支持的消息类型不能为空");
        this.supportedTypes = Arrays.asList(supportedTypes);
    }

    @Override
    public final void handleMessage(Message<?> message) throws MessagingException {
        final MessageHeaders headers = message.getHeaders();
        final String messageType = headers.get(ExsMessageHeaders.TYPE, String.class);
        if (supports(messageType)) {
            handleMessageInternal(message);
        } else {
            log.debug("当前消息处理器「{}」不支持该类型消息的处理：{}", supportedTypes, messageType);
        }
    }

    protected boolean supports(String messageType) {
        return supportedTypes.contains(messageType);
    }

    protected abstract void handleMessageInternal(Message<?> message) throws MessagingException;
}
