package org.opoo.ootp.client.messaging;

import org.springframework.beans.factory.config.ConfigurableBeanFactory;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ConfigurableApplicationContext;
import org.springframework.core.MethodParameter;
import org.springframework.core.annotation.AnnotatedElementUtils;
import org.springframework.core.convert.ConversionService;
import org.springframework.format.support.DefaultFormattingConversionService;
import org.springframework.lang.Nullable;
import org.springframework.messaging.Message;
import org.springframework.messaging.converter.MessageConverter;
import org.springframework.messaging.handler.annotation.MessageMapping;
import org.springframework.messaging.handler.annotation.support.AnnotationExceptionHandlerMethodResolver;
import org.springframework.messaging.handler.annotation.support.DestinationVariableMethodArgumentResolver;
import org.springframework.messaging.handler.annotation.support.HeaderMethodArgumentResolver;
import org.springframework.messaging.handler.annotation.support.HeadersMethodArgumentResolver;
import org.springframework.messaging.handler.annotation.support.MessageMethodArgumentResolver;
import org.springframework.messaging.handler.annotation.support.PayloadMethodArgumentResolver;
import org.springframework.messaging.handler.invocation.AbstractExceptionHandlerMethodResolver;
import org.springframework.messaging.handler.invocation.AbstractMethodMessageHandler;
import org.springframework.messaging.handler.invocation.HandlerMethodArgumentResolver;
import org.springframework.messaging.handler.invocation.HandlerMethodReturnValueHandler;
import org.springframework.messaging.simp.annotation.support.PrincipalMethodArgumentResolver;
import org.springframework.stereotype.Component;
import org.springframework.stereotype.Controller;
import org.springframework.validation.Validator;

import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Objects;
import java.util.Set;

public class MethodExsMessageHandler extends AbstractMethodMessageHandler<MessageMapping> {
    public static final String DEFAULT_DESTINATION = "default";

    private MessageConverter messageConverter;

    private ConversionService conversionService = new DefaultFormattingConversionService();

    @Nullable
    private Validator validator;

    public MessageConverter getMessageConverter() {
        return messageConverter;
    }

    public void setMessageConverter(MessageConverter messageConverter) {
        this.messageConverter = messageConverter;
    }

    public ConversionService getConversionService() {
        return conversionService;
    }

    public void setConversionService(ConversionService conversionService) {
        this.conversionService = conversionService;
    }

    @Nullable
    public Validator getValidator() {
        return validator;
    }

    public void setValidator(@Nullable Validator validator) {
        this.validator = validator;
    }

    @Override
    protected List<? extends HandlerMethodArgumentResolver> initArgumentResolvers() {
        ApplicationContext context = getApplicationContext();
        ConfigurableBeanFactory beanFactory = (context instanceof ConfigurableApplicationContext ?
                ((ConfigurableApplicationContext) context).getBeanFactory() : null);

        List<HandlerMethodArgumentResolver> resolvers = new ArrayList<>();

        // Annotation-based argument resolution
        resolvers.add(new HeaderMethodArgumentResolver(this.conversionService, beanFactory));
        resolvers.add(new HeadersMethodArgumentResolver());
        resolvers.add(new DestinationVariableMethodArgumentResolver(this.conversionService));

        // Type-based argument resolution
        resolvers.add(new PrincipalMethodArgumentResolver());
        resolvers.add(new MessageMethodArgumentResolver(this.messageConverter));

        resolvers.addAll(getCustomArgumentResolvers());
        resolvers.add(new PayloadMethodArgumentResolver(this.messageConverter, this.validator));

        return resolvers;
    }

    @Override
    protected List<? extends HandlerMethodReturnValueHandler> initReturnValueHandlers() {
        return Collections.singletonList(new HandlerMethodReturnValueHandler() {
            @Override
            public boolean supportsReturnType(MethodParameter returnType) {
                return true;
            }

            @Override
            public void handleReturnValue(Object returnValue, MethodParameter returnType, Message<?> message) throws Exception {
                //no op
            }
        });
    }

    @Override
    protected boolean isHandler(Class<?> beanType) {
        return AnnotatedElementUtils.hasAnnotation(beanType, Component.class)
                || AnnotatedElementUtils.hasAnnotation(beanType, Controller.class);
    }

    @Override
    protected MessageMapping getMappingForMethod(Method method, Class<?> handlerType) {
        MessageMapping messageAnn = AnnotatedElementUtils.findMergedAnnotation(method, MessageMapping.class);
        if (messageAnn != null) {
            if (messageAnn.value().length > 0) {
                return messageAnn;
            }
        }
        return null;
    }

    @Override
    protected Set<String> getDirectLookupDestinations(MessageMapping mapping) {
        return Collections.singleton(DEFAULT_DESTINATION);
    }

    @Override
    protected String getDestination(Message<?> message) {
        return DEFAULT_DESTINATION;
    }

    @Override
    protected MessageMapping getMatchingMapping(MessageMapping mapping, Message<?> message) {
        final String type = Objects.requireNonNull(message.getHeaders().get("type", String.class));
        if (Arrays.asList(mapping.value()).contains(type)) {
            return mapping;
        }
        return null;
    }

    @Override
    protected Comparator<MessageMapping> getMappingComparator(Message<?> message) {
        //return ((o1, o2) -> {o1.value().});
        return null;
    }

    @Override
    protected AbstractExceptionHandlerMethodResolver createExceptionHandlerMethodResolverFor(Class<?> beanType) {
        return new AnnotationExceptionHandlerMethodResolver(beanType);
    }
}
