package org.opoo.ootp.client.messaging;

import lombok.extern.slf4j.Slf4j;
import org.opoo.ootp.client.ExsMessage;
import org.opoo.ootp.client.ExsMessageLite;
import org.opoo.ootp.client.MessageClient;
import org.opoo.ootp.client.PollRequest;
import org.opoo.ootp.client.PollResult;
import org.springframework.messaging.Message;
import org.springframework.messaging.SubscribableChannel;
import org.springframework.messaging.converter.MessageConverter;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

@Slf4j
public class DefaultExsMessageDispatcher implements PollableExsMessageDispatcher {
    private final SubscribableChannel subscribableChannel;
    private final MessageConverter messageConverter;
    private final MessageClient messageClient;

    private String clientId;
    private int defaultLimit = 10;
    private boolean ignoreFailures = true;

    private final List<Class<? extends Exception>> returnMessageExceptionTypes = new ArrayList<>();

    public DefaultExsMessageDispatcher(SubscribableChannel subscribableChannel, MessageConverter messageConverter, MessageClient messageClient) {
        this.subscribableChannel = subscribableChannel;
        this.messageConverter = messageConverter;
        this.messageClient = messageClient;
        this.addReturnMessageExceptionTypes(ReturnMessageException.class);
    }

    public String getClientId() {
        return clientId;
    }

    public void setClientId(String clientId) {
        this.clientId = clientId;
    }

    public int getDefaultLimit() {
        return defaultLimit;
    }

    public void setDefaultLimit(int defaultLimit) {
        this.defaultLimit = defaultLimit;
    }

    public boolean isIgnoreFailures() {
        return ignoreFailures;
    }

    public void setIgnoreFailures(boolean ignoreFailures) {
        this.ignoreFailures = ignoreFailures;
    }

    public List<Class<? extends Exception>> getReturnMessageExceptionTypes() {
        return returnMessageExceptionTypes;
    }

    public void addReturnMessageExceptionTypes(Class<? extends Exception> returnMessageExceptionType) {
        this.returnMessageExceptionTypes.add(returnMessageExceptionType);
    }

    @Override
    public void dispatch(ExsMessage exsMessage) {
        final Message<?> toMessage = messageConverter.toMessage(exsMessage, new ExsMessageHeaders());
        subscribableChannel.send(Objects.requireNonNull(toMessage, "转换后的消息不能为空"));

//        final Message<?> toMessage;
//
//        try {
//            toMessage = messageConverter.toMessage(exsMessage, new ExsMessageHeaders());
//        } catch (RuntimeException e) {
//            if (isIgnoreFailures()) {
//                log.warn("消息转换失败，转换后为空：{}", exsMessage);
//                return;
//            } else {
//                throw e;
//            }
//        }
//
//        if (toMessage == null) {
//            if (isIgnoreFailures()) {
//                log.warn("消息转换失败，转换后为空：{}", exsMessage);
//                return;
//            } else {
//                throw new IllegalStateException("消息转换后为空：" + exsMessage);
//            }
//        }
//
//        try {
//            subscribableChannel.send(toMessage);
//        } catch (RuntimeException e) {
//            if (!this.isIgnoreFailures()) {
//                if (e instanceof MessagingException && ((MessagingException) e).getFailedMessage() == null) {
//                    throw new MessagingException(toMessage, "Failed to handle message", e);
//                }
//                throw e;
//            } else {
//                log.warn("消息分发失败。Suppressing Exception since 'ignoreFailures' is set to TRUE.", e);
//            }
//        }
    }

    @Override
    public void dispatch(List<ExsMessage> messages) {
        for (ExsMessage message : messages) {
            dispatch(message);
        }
    }

    @Override
    public void pollAndDispatch(int limit) {
        pollAndDispatch(limit, null);
    }

    @Override
    public void pollAndDispatch() {
        pollAndDispatch(defaultLimit, null);
    }

    private void pollAndDispatch(int limit, String latestId) {
        final PollRequest pollRequest = new PollRequest(limit, true, false, latestId, clientId);
        PollResult pollResult;
        try {
            pollResult = messageClient.poll(pollRequest);
        } catch (Exception e) {
            log.error("轮询时出错，忽略本次轮询结果", e);
            return;
        }

        if (pollResult == null) {
            log.warn("轮询结果为 null");
            return;
        }

        final List<ExsMessageLite> messages = pollResult.getMessages();
        final int size = messages.size();
        log.debug("轮询消息 {} 条", size);
        messages.forEach(this::dispatch);

//        final List<ExsMessage> exsMessages = messages.stream()
//                .map(lite -> messageClient.getMessage(lite.getMetadata().getId()))
//                .peek(message -> messageClient.ack(message.getMetadata().getId()))
//                .collect(Collectors.toList());
//
//        dispatch(exsMessages);

        if (pollResult.hasNext()) {
            pollAndDispatch(limit, pollRequest.getLatestId());
        }
    }

    public void dispatch(ExsMessageLite lite) {
        final String messageId = lite.getMetadata().getId();
        try {
            dispatch(messageClient.getMessage(messageId));
//        } catch (ReturnMessageException e) {
//            log.warn("捕获 ReturnMessageException，退回消息：{}", messageId, e);
//            tryReturnMessage(messageId);
        } catch (Exception e) {
            if (returnMessageExceptionTypes.stream().anyMatch(type -> type.isInstance(e))) {
                log.warn("消息处理出错，根据配置，发生此异常将回退消息：{} - {}", messageId, e);
                tryReturnMessage(messageId);
                return;
            }

            if (isIgnoreFailures()) {
                log.warn("消息处理失败，忽略错误：" + e.getMessage(), e);
                return;
            }

            throw e;
        }
    }

    private void tryReturnMessage(String messageId) {
        try {
            final int rows = messageClient.ret(messageId);
            log.info("消息退回：{}", rows);
        } catch (Exception e) {
            log.error("消息退回失败：{}", messageId, e);
        }
    }

}
