package org.opoo.ootp.signer.httpclient;

import lombok.extern.slf4j.Slf4j;
import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.HttpRequestInterceptor;
import org.apache.http.NameValuePair;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.client.utils.URLEncodedUtils;
import org.apache.http.entity.BufferedHttpEntity;
import org.apache.http.protocol.HttpContext;
import org.opoo.ootp.signer.Signer;

import java.io.IOException;
import java.io.InputStream;
import java.util.Arrays;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import static java.nio.charset.StandardCharsets.UTF_8;

@Slf4j
public class SignerHttpRequestInterceptor implements HttpRequestInterceptor {
    private final Signer signer;

    public SignerHttpRequestInterceptor(Signer signer) {
        this.signer = signer;
    }

    @Override
    public void process(HttpRequest request, HttpContext context) throws HttpException, IOException {
        if (request instanceof HttpUriRequest) {
            sign((HttpUriRequest) request, context);
        } else {
            log.warn("Not HttpUriRequest, skip sign the request: {}", request);
        }
    }

    protected void sign(HttpUriRequest request, HttpContext context) throws HttpException, IOException {
        final Map<String, String> params = new LinkedHashMap<>();
        final Map<String, String> headers = new LinkedHashMap<>();

        extractContentType(request);
        extractParameters(request, params);

        Arrays.stream(request.getAllHeaders()).forEach(header -> headers.put(header.getName(), header.getValue()));

        // 如果请求里提供了，则用之，若无则算之
        final Optional<String> contentHashOpt = Optional.ofNullable(request.getFirstHeader(Signer.HEADER_NAME_CONTENT_HASH)).map(Header::getValue);
        log.debug("是否需要计算内容的哈希：{}", !contentHashOpt.isPresent());
        final String contentHash = contentHashOpt.isPresent() ? contentHashOpt.get() : calculateContentHash(request);

        final String authorization = signer.buildAuthorization(
                request.getMethod(),
                request.getURI().getPath(),
                headers,
                params,
                contentHash);

        log.debug("Auth：{}", authorization);
        request.setHeader("Authorization", authorization);
    }

    private void extractContentType(HttpUriRequest request) {
        if (request instanceof HttpEntityEnclosingRequest && !request.containsHeader("Content-Type")) {
            HttpEntity entity = ((HttpEntityEnclosingRequest) request).getEntity();
            Header contentType = entity != null ? entity.getContentType() : null;
            String value = contentType != null ? contentType.getValue() : null;

            if (value != null) {
                request.addHeader(contentType);
            }
        }
    }

    private void extractParameters(HttpUriRequest request, Map<String, String> params) throws IOException {
        // Query String
        List<NameValuePair> list = URLEncodedUtils.parse(request.getURI(), UTF_8);
        for (NameValuePair pair : list) {
            params.put(pair.getName(), pair.getValue());
        }

        // 表单
        if (request instanceof HttpEntityEnclosingRequest) {
            HttpEntity entity = ((HttpEntityEnclosingRequest) request).getEntity();
            if (entity instanceof UrlEncodedFormEntity) {
                List<NameValuePair> list2 = URLEncodedUtils.parse(entity);
                for (NameValuePair pair : list2) {
                    params.put(pair.getName(), pair.getValue());
                }
            }
        }
    }

    private String calculateContentHash(HttpUriRequest request) throws IOException {
        InputStream content = null;
        if (request instanceof HttpEntityEnclosingRequest) {
            HttpEntity entity = ((HttpEntityEnclosingRequest) request).getEntity();
            if (entity != null) {
                if (!entity.isRepeatable()) {
                    log.debug("Entity is not repeatable, create buffered entity.");
                    entity = new BufferedHttpEntity(entity);
                    ((HttpEntityEnclosingRequest) request).setEntity(entity);
                }

                content = entity.getContent();
            }
        }

        return signer.hash(content);
    }
}
