package org.opoo.ootp.signer.spring;

import lombok.extern.slf4j.Slf4j;
import org.opoo.ootp.signer.Signer;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpRequest;
import org.springframework.http.client.ClientHttpRequestExecution;
import org.springframework.http.client.ClientHttpRequestInterceptor;
import org.springframework.http.client.ClientHttpResponse;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;
import org.springframework.web.util.UriComponentsBuilder;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;

@Slf4j
public class SignerClientHttpRequestInterceptor implements ClientHttpRequestInterceptor {
    private final Signer signer;

    public SignerClientHttpRequestInterceptor(Signer signer) {
        this.signer = signer;
    }

    @Override
    public ClientHttpResponse intercept(HttpRequest request, byte[] body, ClientHttpRequestExecution execution) throws IOException {
        final MultiValueMap<String, String> params = new LinkedMultiValueMap<>();
        extractParameters(request, params);

        final HttpHeaders headers = request.getHeaders();
        // 如果请求里提供了，则用之，若无则算之
        final Optional<String> contentHashOpt = Optional.ofNullable(headers.getFirst(Signer.HEADER_NAME_CONTENT_HASH));
        log.debug("是否需要计算内容的哈希：{}", !contentHashOpt.isPresent());
        final String contentHash = contentHashOpt.isPresent() ? contentHashOpt.get() : calculateContentHash(body);

        final String authorization = signer.buildAuthorization(
                Optional.ofNullable(request.getMethod()).map(Enum::name).orElse(""),
                request.getURI().getPath(),
                headers.toSingleValueMap(),
                params.toSingleValueMap(),
                contentHash);

        log.debug("Auth：{}", authorization);
        headers.set("Authorization", authorization);

        return execution.execute(request, body);
    }

    private void extractParameters(HttpRequest request, MultiValueMap<String,String> params){
        MultiValueMap<String, String> queryParams = UriComponentsBuilder.fromUri(request.getURI()).build().getQueryParams();
        for(Map.Entry<String, List<String>> entry: queryParams.entrySet()){
            params.put(entry.getKey(), decodeStringList(entry.getValue()));
        }
    }

    private List<String> decodeStringList(List<String> list){
        List<String> result = new ArrayList<>();
        for(String string: list){
            if (string == null) {
                result.add(null);
                continue;
            }

            try {
                result.add(URLDecoder.decode(string, "UTF-8"));
            } catch (UnsupportedEncodingException ex) {
                if (log.isDebugEnabled()) {
                    log.error("URL 解码失败，忽略解码结果：" + string, ex);
                }
                result.add(string);
            }
        }
        return result;
    }

    private String calculateContentHash(byte[] body) throws IOException {
        ByteArrayInputStream inputStream = body != null && body.length > 0 ? new ByteArrayInputStream(body) : null;
        return signer.hash(inputStream);
    }
}