package org.opoo.ootp.signer;

import lombok.extern.slf4j.Slf4j;

import java.time.Instant;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.AbstractMap;
import java.util.Map;
import java.util.stream.Collectors;

@Slf4j
public abstract class AbstractSigner implements Signer {

    /**
     * format strings for the date/time and date stamps required during signing
     */
    public static String ISO8601_FORMAT = "yyyyMMdd'T'HHmmss'Z'";
    public static DateTimeFormatter GMT = DateTimeFormatter.ofPattern(ISO8601_FORMAT).withZone(ZoneId.of("GMT"));

    private final String accessKey;
    private final String secretKey;

    public AbstractSigner(String accessKey, String secretKey) {
        this.accessKey = accessKey;
        this.secretKey = secretKey;
    }

    public String getAccessKey() {
        return accessKey;
    }

    public String getSecretKey() {
        return secretKey;
    }

    protected abstract String getAlgorithm();

    @Override
    public String buildAuthorization(String httpMethod, String resourcePath,
                                     Map<String, String> headers,
                                     Map<String, String> parameters,
                                     String contentHash) {
        final String dateString = GMT.format(Instant.now());
        final String stringToSign = buildStringToSign(getAlgorithm(), dateString, httpMethod, resourcePath, headers, parameters, contentHash);
        log.debug("StringToSign: [{}]", stringToSign);
        final String signature = sign(stringToSign);
        return SCHEME + "-" + getAlgorithm() + " " + accessKey + ":" + dateString + ":" + signature;
    }

    public String buildStringToSign(String algorithm,
                                    String dateTime,
                                    String httpMethod, String resourcePath,
                                    Map<String, String> headers,
                                    Map<String, String> parameters,
                                    String contentHash) {
        return SCHEME + "-" + algorithm + "\n" +
                dateTime + "\n" +
                httpMethod + "\n" +
                resourcePath + "\n" +
                getCanonicalizedHeaderString(headers) + "\n" +
                getCanonicalizedQueryString(parameters) + "\n" +
                contentHash;
    }

    public String getCanonicalizedHeaderString(Map<String, String> headers) {
        if (headers == null || headers.isEmpty()) {
            return "";
        }

        return headers.entrySet().stream()
                // key to lower case
                .map(entry -> new AbstractMap.SimpleEntry<>(entry.getKey().toLowerCase(), entry.getValue()))
                // filter: only 'content-type' and the headers name starts with 'x-xxx-'
                .filter(entry -> "content-type".equalsIgnoreCase(entry.getKey()) || entry.getKey().startsWith(HEADER_NAME_PREFIX))
                // sort the headers by case-insensitive order
                .sorted(Map.Entry.comparingByKey())
                // form the canonical header:value entries in sorted order.
                // Multiple white spaces in the values should be compressed to a single space.
                .map(entry -> entry.getKey().replaceAll("\\s+", " ") + ":" +
                        entry.getValue().replaceAll("\\s+", " ") + "\n")
                .collect(Collectors.joining(""));
    }

    public String getCanonicalizedQueryString(Map<String, String> parameters) {
        if (parameters == null || parameters.isEmpty()) {
            return "";
        }

        return parameters.entrySet().stream()
                // sort the parameters by case-sensitive order
                .sorted(Map.Entry.comparingByKey())
                // form the canonical name1=value1&amp;name2=value2 entries in sorted order.
                .map(entry -> entry.getKey() + "=" + entry.getValue())
                .collect(Collectors.joining("&"));
    }

}
