package org.opoo.tools.db;

import lombok.Data;
import lombok.ToString;
import org.springframework.util.Assert;
import org.springframework.util.LinkedCaseInsensitiveMap;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

@Data
public class Table {
    /**
     * 表名
     */
    private final String name;
    /**
     * 必须按主键、非主键字段的顺序排列。
     */
    private final Column[] columns;

    /**
     * 基本的 WHERE 条件，用于对比或对比部分满足条件的数据。
     * where_condition
     */
    private String whereCondition;

    @ToString.Exclude
    private final Map<String, Column> columnsMap = new LinkedCaseInsensitiveMap<>();
    @ToString.Exclude
    private final Column[] primaryKeyColumns;
    @ToString.Exclude
    private final Column[] normalColumns;

    public Table(String name, Column... columns) {
        Assert.isTrue(columns.length > 0, "字段数必须大于 0");

        this.name = name;

        final List<Column> primaryKeyColumnList = new ArrayList<>();
        final List<Column> normalColumnList = new ArrayList<>();

        for (Column column : columns) {
            column.setTable(this);
            // 注意，key 可能是个表达式
            columnsMap.put(column.getName(), column);
            if (column.isPrimaryKey()) {
                primaryKeyColumnList.add(column);
            } else {
                normalColumnList.add(column);
            }
        }

        Assert.isTrue(!primaryKeyColumnList.isEmpty(), "主键字段数必须大于 0");

        this.primaryKeyColumns = primaryKeyColumnList.toArray(new Column[0]);
        this.normalColumns = normalColumnList.toArray(new Column[0]);
        this.columns = new Column[columns.length];
        System.arraycopy(primaryKeyColumns, 0, this.columns, 0, primaryKeyColumns.length);
        System.arraycopy(normalColumns, 0, this.columns, primaryKeyColumns.length, normalColumns.length);
    }

    public Table withWhereCondition(String whereCondition) {
        this.whereCondition = whereCondition;
        return this;
    }

}

