package org.opoo.tools.db.copy;

import lombok.Setter;
import lombok.extern.slf4j.Slf4j;
import org.opoo.tools.db.Column;
import org.opoo.tools.db.Id;
import org.opoo.tools.db.SqlAndParams;
import org.opoo.tools.db.Table;
import org.opoo.tools.db.util.DbUtils;
import org.springframework.lang.Nullable;
import org.springframework.util.StringUtils;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;


@Setter
@Slf4j
public class MySQLSameDbTableCopier extends AbstractSameDbTableCopier {
    private boolean usingOnDuplicateKeyUpdate = false;

    @Override
    protected SqlAndParams buildGetOffsetIdSql(Table table, int batchSize, Id previousOffsetId) {
        final String[] primaryKeyNames = Arrays.stream(table.getPrimaryKeyColumns()).map(Column::getName).toArray(String[]::new);
        final String primaryKeyJoinString = String.join(", ", primaryKeyNames);
        final String whereCondition = table.getWhereCondition();
        final List<Object> params = new ArrayList<>();

        final StringBuilder sql = new StringBuilder("SELECT " + primaryKeyJoinString + " FROM " + table.getName());
        if (previousOffsetId != null) {
            final SqlAndParams idCondition = DbUtils.buildGreaterThanCondition(primaryKeyNames, previousOffsetId.getValues());
            sql.append(" WHERE ").append(idCondition.getSql());
            params.addAll(idCondition.getParams());
        }

        if (StringUtils.hasText(whereCondition)) {
            if (previousOffsetId != null) {
                sql.append(" AND (").append(whereCondition).append(")");
            } else {
                sql.append(" WHERE ").append(whereCondition);
            }
        }
        // MySQL/MariaDB/TDSQl for MySQL 使用 limit offset 获取特定记录
        sql.append(" ORDER BY ").append(primaryKeyJoinString).append(" LIMIT 1 OFFSET ?");
        params.add(batchSize - 1);

        return new SqlAndParams(sql.toString(), params);
    }

    @Override
    protected SqlAndParams buildCopySql(Table sourceTable, Table targetTable, @Nullable Id previousOffsetId, @Nullable Id offsetId) {
        // MySQL/MariaDB/TDSQl for MySQL 使用 ON DUPLICATED KEY UPDATE 的方式实现增量复制
        final SqlAndParams sqlAndParams = super.buildCopySql(sourceTable, targetTable, previousOffsetId, offsetId);
        if (!usingOnDuplicateKeyUpdate) {
            return sqlAndParams;
        }

        final StringBuilder sql = new StringBuilder(sqlAndParams.getSql());
        final List<Object> params = sqlAndParams.getParams();

        sql.append(" ON DUPLICATE KEY UPDATE ");
        final Column[] normalColumnsB = targetTable.getNormalColumns();
        final int size = normalColumnsB.length;
        for (int i = 0; i < size; i++) {
            sql.append(normalColumnsB[i].getName()).append(" = VALUES(").append(normalColumnsB[i].getName()).append(")");
            if (i < size - 1) {
                sql.append(", ");
            }
        }

        return new SqlAndParams(sql.toString(), params);
    }
}
