package org.opoo.tools.db.diff;

import lombok.extern.slf4j.Slf4j;
import org.opoo.tools.db.Database;
import org.opoo.tools.db.DatabaseInput;
import org.opoo.tools.db.TableNameInput;
import org.opoo.tools.db.util.DbUtils;
import org.opoo.tools.db.util.ListUtils;

import java.sql.SQLException;
import java.util.List;

@Slf4j
public class SimpleDatabaseComparator implements DatabaseComparator {
    private final TableComparator tableComparator;

    public SimpleDatabaseComparator() {
        this.tableComparator = new SimpleTableComparator();
    }

    public SimpleDatabaseComparator(TableComparator tableComparator) {
        this.tableComparator = tableComparator;
    }

    @Override
    public void compare(DatabaseInput dbA, DatabaseInput dbB, DatabaseListener listener) throws SQLException {
        final Database databaseA = dbA.getDatabase();
        final Database databaseB = dbB.getDatabase();
        final List<String> tableNamesA = DbUtils.getTableNames(databaseA.getCatalog(), databaseA.getTableNamePattern(),
                databaseA.getTableNamePattern(), dbA.getConnection());
        final List<String> tableNamesB = DbUtils.getTableNames(databaseB.getCatalog(), databaseB.getTableNamePattern(),
                databaseB.getTableNamePattern(), dbB.getConnection());

        final List<String> tableNamesOnlyInDbA = ListUtils.removeAllIgnoreCase(tableNamesA, tableNamesB);
        final List<String> tableNamesOnlyInDbB = ListUtils.removeAllIgnoreCase(tableNamesB, tableNamesA);
        final List<String> tableNames = ListUtils.intersectionIgnoreCase(tableNamesA, tableNamesB);

        tableNamesOnlyInDbA.forEach(listener::onOnlyInDbA);
        tableNamesOnlyInDbB.forEach(listener::onOnlyInDbB);

        // 两边都有的表
        for (String tableName : tableNames) {
            final TableResult result = TableResult.arrayListResult(tableName + "@A", tableName + "@B");
            final TableResultListener resultListener = new TableResultListener(result);
            try {
                tableComparator.compare(
                        new TableNameInput(tableName, dbA.getConnection()),
                        new TableNameInput(tableName, dbB.getConnection()),
                        resultListener
                );
            } catch (Exception ex) {
                listener.onTableException(tableName, ex);
            }
            if (result.getIdsDifferent().isEmpty() && result.getIdsOnlyInA().isEmpty() && result.getIdsOnlyInB().isEmpty()) {
                listener.onIdentical(tableName);
            } else {
                listener.onTableDifferent(tableName, result);
            }
        }
    }

}
