package org.opoo.tools.db.diff;

import org.opoo.tools.db.SqlEqualizer;
import org.opoo.tools.db.Table;
import org.opoo.tools.db.TableInput;
import org.opoo.tools.db.TableNameInput;
import org.opoo.tools.db.util.DbUtils;

import java.sql.ResultSet;
import java.sql.SQLException;

/**
 * 表数据对比器。
 */
public interface TableComparator {

    /**
     * 比对两个表的数据。
     *
     * @param tableInputA 表A，必要参数
     * @param tableInputB 表B，必要参数
     * @param equalizer   个性化记录比较器，可选参数
     * @param listener    对比结果监听器，必要参数
     * @throws SQLException 可能出现的异常
     */
    void compare(TableInput tableInputA, TableInput tableInputB, SqlEqualizer<ResultSet> equalizer, TableListener listener)
            throws SQLException;

    default void compare(TableInput tableInputA, TableInput tableInputB, TableListener listener) throws SQLException {
        compare(tableInputA, tableInputB, null, listener);
    }

    /**
     * 比对两个表的数据。 根据表名自动从数据库的 meta 中获取表的列信息，构建 {@link Table} 对象，然后再比对。
     *
     * @param tableInputA 表A的名称和数据库连接
     * @param tableInputB 表B的名称和数据库连接
     * @param equalizer   个性化记录比较器
     * @param listener    比对结果监听器
     * @throws SQLException SQL操作出现的异常
     */
    default void compare(TableNameInput tableInputA, TableNameInput tableInputB, SqlEqualizer<ResultSet> equalizer, TableListener listener)
            throws SQLException {
        final Table a = DbUtils.buildTable(tableInputA.getTableName(), tableInputA.getConnection());
        final Table b = DbUtils.buildTable(tableInputB.getTableName(), tableInputB.getConnection());
        final TableInput inputA = new TableInput(a, tableInputA.getConnection());
        final TableInput inputB = new TableInput(b, tableInputB.getConnection());
        compare(inputA, inputB, equalizer, listener);
    }

    default void compare(TableNameInput tableInputA, TableNameInput tableInputB, TableListener listener) throws SQLException {
        compare(tableInputA, tableInputB, null, listener);
    }
}
