package org.opoo.tools.db.util;

import lombok.experimental.UtilityClass;
import lombok.extern.slf4j.Slf4j;
import org.opoo.tools.db.Column;
import org.opoo.tools.db.Id;
import org.opoo.tools.db.SqlAndParams;
import org.opoo.tools.db.SqlSupplier;
import org.opoo.tools.db.Table;
import org.springframework.jdbc.support.JdbcUtils;

import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;

@Slf4j
@UtilityClass
public class DbUtils {

    /**
     * 获取下一条记录的主键值，如果已经到表尾，返回null。
     */
    public static Id getNextId(Table table, ResultSet resultSet) throws SQLException {
        return getNextId(table.getPrimaryKeyColumns(), resultSet);
    }

    public static Id getNextId(Column[] primaryKeys, ResultSet resultSet) throws SQLException {
        if (resultSet.next()) {
            return getId(primaryKeys, resultSet);
        }
        return null;
    }

    public static Id getId(Column[] primaryKeys, ResultSet resultSet) throws SQLException {
        final Object[] values = new Object[primaryKeys.length];
        for (int i = 0; i < primaryKeys.length; i++) {
            values[i] = JdbcUtils.getResultSetValue(resultSet, i + 1, primaryKeys[i].getType());
        }
        return new Id(values);
    }

    /**
     * 对指定的字段集合的字段类型进行初始化。
     *
     * @param rs      ResultSet 中的字段顺序应该与 columns 的顺序一致，且数量大于等于 columns
     * @param columns 字段的顺序应该与 ResultSet 的一致，且数量小于等于 ResultSet
     * @throws SQLException 可能出现的错误
     */
    public static void initializeColumnTypes(ResultSet rs, Column[] columns) throws SQLException {
        // 优先判断，省掉创建 ResultMetaData 的消耗，但是判断也有消耗，那个更大？？
        final boolean allMatch = Arrays.stream(columns).allMatch(c -> Objects.nonNull(c.getType()));
        if (allMatch) {
            log.debug("对应字段都已经设置了类型，不必再处理");
            return;
        }

        final ResultSetMetaData resultSetMetaData = rs.getMetaData();
        for (int i = 0; i < columns.length; i++) {
            final Column column = columns[i];
            // 仅对用户没有指定类型的才赋值
            if (Objects.isNull(column.getType())) {
                final int columnType = resultSetMetaData.getColumnType(i + 1);
                column.setSqlType(columnType);

                final Class<?> type = SqlTypeUtils.sqlTypeToClass(columnType);
                column.setType(type);

                if (log.isDebugEnabled()) {
                    log.debug("表[{}]列[{}]解析值类型：{}/{}", column.getTable().getName(), column.getName(), columnType, type);
                }
            }
        }
    }

    public static <T> T orElse(T obj, SqlSupplier<T> supplier) throws SQLException {
        if (obj == null) {
            return supplier.get();
        }
        return obj;
    }

    /**
     * 获取catalog，获取失败返回{@code null}
     *
     * @param conn {@link Connection} 数据库连接，{@code null} 时返回null
     * @return catalog，获取失败返回{@code null}
     */
    public static String getCatalog(Connection conn) {
        if (null == conn) {
            return null;
        }
        try {
            return conn.getCatalog();
        } catch (SQLException e) {
            // ignore
        }

        return null;
    }

    /**
     * 获取schema，获取失败返回{@code null}
     *
     * @param conn {@link Connection} 数据库连接，{@code null}时返回null
     * @return schema，获取失败返回{@code null}
     */
    public static String getSchema(Connection conn) {
        if (null == conn) {
            return null;
        }
        try {
            return conn.getSchema();
        } catch (SQLException e) {
            // ignore
        }

        return null;
    }

    /**
     * 获取表的字段集合，含字段是否主键的标识。
     */
    public static Table buildTable(String tableName, Connection conn) throws SQLException {
        return buildTable(getCatalog(conn), getSchema(conn), tableName, conn);
    }

    public static Table buildTable(String catalog, String schemaPattern, String tableName, Connection conn) throws SQLException {
        final DatabaseMetaData metaData = conn.getMetaData();
        final List<String> primaryKeys = new ArrayList<>();

        try (final ResultSet rs = metaData.getPrimaryKeys(getCatalog(conn), getSchema(conn), tableName)) {
            while (rs.next()) {
                primaryKeys.add(rs.getString("COLUMN_NAME"));
            }
        }

        final List<Column> columnList = new ArrayList<>();
        try (final ResultSet columns = metaData.getColumns(catalog, schemaPattern, tableName, null)) {
            while (columns.next()) {
                final String columnName = columns.getString("COLUMN_NAME");
                final boolean isPrimaryKey = primaryKeys.contains(columnName);
                final int dataType = columns.getInt("DATA_TYPE");
                final Class<?> type = SqlTypeUtils.sqlTypeToClass(dataType);

                final Column column = new Column(columnName, isPrimaryKey).withType(type).withSqlType(dataType);
                columnList.add(column);
            }
        }

        log.debug("table {}: {}", tableName, columnList);
        return new Table(tableName, columnList.toArray(new Column[0]));
    }

    public static List<String> getTableNames(Connection conn) throws SQLException {
        return getTableNames(getCatalog(conn), getSchema(conn), "%", conn);
    }

    public static List<String> getTableNames(String catalog, String schemaPattern, String tableNamePattern, Connection conn) throws SQLException {
        final List<String> tableNames = new ArrayList<>();
        final DatabaseMetaData metaData = conn.getMetaData();
        try (final ResultSet tables = metaData.getTables(catalog, schemaPattern, tableNamePattern, new String[]{"TABLE"})) {
            while (tables.next()) {
                final String tableName = tables.getString("TABLE_NAME");
                tableNames.add(tableName);
            }
        }
        return tableNames;
    }

    public static SqlAndParams buildGreaterThanCondition(String[] columnNames, Object[] values) {
        return buildCondition(columnNames, values, true, false);
    }

    public static SqlAndParams buildLessThanCondition(String[] columnNames, Object[] values) {
        return buildCondition(columnNames, values, false, false);
    }

    public static SqlAndParams buildGreaterThanOrEqualsCondition(String[] columnNames, Object[] values) {
        return buildCondition(columnNames, values, true, true);
    }

    public static SqlAndParams buildLessThanOrEqualsCondition(String[] columnNames, Object[] values) {
        return buildCondition(columnNames, values, false, true);
    }

    public static SqlAndParams buildCondition(String[] columnNames, Object[] values, boolean greaterThan, boolean includeEquals) {
        final List<Object> params = new ArrayList<>();
        final StringBuilder condition = new StringBuilder();
        final int length = columnNames.length;
        final int maxIndex = length - 1;

        if (length > 1) {
            condition.append("(");
        }

        for (int i = 0; i <= maxIndex; i++) {
            if (i > 0) {
                condition.append(" OR ");
            }
            condition.append("(");
            for (int j = 0; j <= i; j++) {
                if (j > 0) {
                    condition.append(" AND ");
                }
                condition.append(columnNames[j]);
                params.add(values[j]);
                if (j < i) {
                    condition.append(" = ?");
                } else {
                    condition.append(greaterThan ? " >" : " <");
                    condition.append(includeEquals && j == maxIndex ? "=" : "");
                    condition.append(" ?");
                }
            }
            condition.append(")");
        }

        if (length > 1) {
            condition.append(")");
        }

        return new SqlAndParams(condition.toString(), params);
    }

}
