/*
 * Copyright 2013 Alex Lin.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.opoo.press.generator;

import org.opoo.press.*;
import org.opoo.press.impl.AbstractConvertible;
import org.opoo.press.Source;
import org.opoo.util.I18NUtills;
import org.opoo.util.URLUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.util.*;

/**
 * @author Alex Lin
 *
 */
public class CategoryGenerator implements Generator {
	private static final Logger log = LoggerFactory.getLogger(CategoryGenerator.class);
	/* (non-Javadoc)
	 * @see org.opoo.press.Ordered#getOrder()
	 */
	@Override
	public int getOrder() {
		return 200;
	}

	/* (non-Javadoc)
	 * @see org.opoo.press.Generator#generate(org.opoo.press.Site)
	 */
	@Override
	public void generate(Site site) {
		log.debug("Generating category pages...");
		List<Category> categories = site.getCategories();

		String categoryPageTitlePrefix = getCategoryPageTitlePrefix(site);
		String template = getCategoryPageTemplate(site);

		for(Category category: categories){
			List<Post> posts = category.getPosts();
			if(posts.isEmpty()){
				continue;
			}
			Collections.sort(posts);
			Collections.reverse(posts);
			
			CategoryPage page = new CategoryPage(site, template);
			page.setTitle(categoryPageTitlePrefix + category.getTitle());
			page.setUrl(category.getUrl());
			page.setPosts(posts);
			
			site.getPages().add(page);
		}
	}

	private String getCategoryPageTitlePrefix(Site site){
		String prefix = I18NUtills.getString("messages", site.getLocale(), "category.page.title.prefix");
		if(prefix == null){
			prefix = site.getConfig().get("category_page_title_prefix", "");
		}
		return prefix;
	}

	private String getCategoryPageTemplate(Site site) {
		return site.getConfig().get("category_page_template", "category.ftl");
	}


	public static class CategoryPage extends AbstractConvertible implements Page{
//		public static final String TEMPLATE = "category.ftl";
		private String url;
		private Renderer renderer;
		private Site site;
		private String content;// = "<#include \"category_index.ftl\">";
		private String title;
		private List<Post> posts;
		private String template;
		
		public CategoryPage(Site site, String template) {
			super();
			this.site = site;
			this.renderer = site.getRenderer();
			this.template = template;
		}

		@Override
		public void render(Map<String, Object> rootMap) {
			rootMap = new HashMap<String,Object>(rootMap);
			mergeRootMap(rootMap);

			String output = getRenderer().render(template, rootMap);
			setContent(output);
		}

		public String getUrl() {
			return url;
		}

		public void setUrl(String url) {
			this.url = url;
		}
		
		/* (non-Javadoc)
		 * @see org.opoo.press.Base#getSource()
		 */
		@Override
		public Source getSource() {
			return null;
		}

		/* (non-Javadoc)
		 * @see org.opoo.press.Base#getContent()
		 */
		@Override
		public String getContent() {
			return content;
		}

		/* (non-Javadoc)
		 * @see org.opoo.press.Base#getPath()
		 */
		@Override
		public String getPath() {
			return null;
		}

		/* (non-Javadoc)
		 * @see org.opoo.press.Base#getLayout()
		 */
		@Override
		public String getLayout() {
			return "nil";
		}

		/* (non-Javadoc)
		 * @see org.opoo.press.Base#getPermalink()
		 */
		@Override
		public String getPermalink() {
			return null;
		}

		/* (non-Javadoc)
		 * @see org.opoo.press.Base#getDate()
		 */
		@Override
		public Date getDate() {
			return null;
		}

		/* (non-Javadoc)
		 * @see org.opoo.press.Base#getUpdated()
		 */
		@Override
		public Date getUpdated() {
			return null;
		}

		/* (non-Javadoc)
		 * @see org.opoo.press.Base#getDateFormatted()
		 */
		@Override
		public String getDateFormatted() {
			return null;
		}

		/* (non-Javadoc)
		 * @see org.opoo.press.Base#getUpdatedFormatted()
		 */
		@Override
		public String getUpdatedFormatted() {
			return null;
		}

		/* (non-Javadoc)
		 * @see org.opoo.press.Page#getPager()
		 */
		@Override
		public Pager getPager() {
			return null;
		}

		/* (non-Javadoc)
		 * @see org.opoo.press.Page#setPager(org.opoo.press.Pager)
		 */
		@Override
		public void setPager(Pager pager) {
		}

		/* (non-Javadoc)
		 * @see org.opoo.press.impl.AbstractConvertible#setContent(java.lang.String)
		 */
		@Override
		public void setContent(String content) {
			this.content = content;
		}

		/* (non-Javadoc)
		 * @see org.opoo.press.impl.AbstractConvertible#getOutputFileExtension()
		 */
		@Override
		public String getOutputFileExtension() {
			return ".html";
		}

		/* (non-Javadoc)
		 * @see org.opoo.press.impl.AbstractConvertible#getRenderer()
		 */
		@Override
		protected Renderer getRenderer() {
			return renderer;
		}

		/* (non-Javadoc)
		 * @see org.opoo.press.impl.AbstractConvertible#convert()
		 */
		@Override
		public void convert() {
			//do nothing
		}

		/* (non-Javadoc)
		 * @see org.opoo.press.impl.AbstractConvertible#mergeRootMap(java.util.Map)
		 */
		@Override
		protected void mergeRootMap(Map<String, Object> rootMap) {
			rootMap.put("canonical", site.buildCanonical(getUrl()));
			rootMap.put("page", this);
		}

		public String getTitle() {
			return title;
		}

		public void setTitle(String title) {
			this.title = title;
		}

		public List<Post> getPosts() {
			return posts;
		}

		public void setPosts(List<Post> posts) {
			this.posts = posts;
		}
		
		public boolean isFooter(){
			return false;
		}
		
		public boolean isSidebar(){
			return true;
		}
		
		public Object get(String string){
			return null;
		}

		@Override
		public File getOutputFile(File dest) {
			String url = getUrl() + "index.html";
			url = URLUtils.decodeURL(url);
			File target = new File(dest, url);
			return target;
		}
		
		public boolean isComments(){
			return false;
		}

		/* (non-Javadoc)
		 * @see org.opoo.press.Base#set(java.lang.String, java.lang.Object)
		 */
		@Override
		public void set(String name, Object value) {
		}
	}
}
