/*
 * Decompiled with CFR 0.152.
 */
package com.caoccao.javet.interop.engine;

import com.caoccao.javet.enums.JSRuntimeType;
import com.caoccao.javet.exceptions.JavetError;
import com.caoccao.javet.exceptions.JavetException;
import com.caoccao.javet.interfaces.IJavetLogger;
import com.caoccao.javet.interop.V8Host;
import com.caoccao.javet.interop.V8Runtime;
import com.caoccao.javet.interop.engine.IJavetEngine;
import com.caoccao.javet.interop.engine.IJavetEnginePool;
import com.caoccao.javet.interop.engine.JavetEngine;
import com.caoccao.javet.interop.engine.JavetEngineConfig;
import com.caoccao.javet.interop.engine.JavetEngineUsage;
import com.caoccao.javet.interop.engine.observers.IV8RuntimeObserver;
import com.caoccao.javet.interop.monitoring.V8SharedMemoryStatistics;
import com.caoccao.javet.interop.options.RuntimeOptions;
import com.caoccao.javet.interop.options.V8RuntimeOptions;
import com.caoccao.javet.utils.JavetDateTimeUtils;
import java.time.ZonedDateTime;
import java.time.temporal.ChronoUnit;
import java.util.Objects;
import java.util.Random;
import java.util.TreeSet;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;

public class JavetEnginePool<R extends V8Runtime>
implements IJavetEnginePool<R>,
Runnable {
    protected static final String JAVET_DAEMON_THREAD_NAME = "Javet Daemon";
    protected final Object externalLock;
    protected final ConcurrentLinkedQueue<Integer> idleEngineIndexList;
    protected final Object internalLock;
    protected final ConcurrentLinkedQueue<Integer> releasedEngineIndexList;
    protected volatile boolean active;
    protected JavetEngineConfig config;
    protected Thread daemonThread;
    protected JavetEngine<R>[] engines;
    protected volatile boolean quitting;
    protected Random random;
    protected Semaphore semaphore;

    public JavetEnginePool() {
        this(new JavetEngineConfig());
    }

    public JavetEnginePool(JavetEngineConfig config) {
        this.config = Objects.requireNonNull(config).freezePoolSize();
        this.idleEngineIndexList = new ConcurrentLinkedQueue();
        this.releasedEngineIndexList = new ConcurrentLinkedQueue();
        this.engines = new JavetEngine[config.getPoolMaxSize()];
        this.externalLock = new Object();
        this.internalLock = new Object();
        this.active = false;
        this.quitting = false;
        this.random = new Random();
        this.semaphore = null;
        this.startDaemon();
    }

    @Override
    public void close() throws JavetException {
        this.stopDaemon();
    }

    protected JavetEngine<R> createEngine() throws JavetException {
        JSRuntimeType jsRuntimeType = this.config.getJSRuntimeType();
        Object runtimeOptions = jsRuntimeType.getRuntimeOptions();
        if (runtimeOptions instanceof V8RuntimeOptions) {
            V8RuntimeOptions v8RuntimeOptions = (V8RuntimeOptions)runtimeOptions;
            v8RuntimeOptions.setGlobalName(this.config.getGlobalName());
        }
        Object v8Runtime = V8Host.getInstance(jsRuntimeType).createV8Runtime(true, (RuntimeOptions<?>)runtimeOptions);
        ((V8Runtime)v8Runtime).allowEval(this.config.isAllowEval());
        ((V8Runtime)v8Runtime).setLogger(this.config.getJavetLogger());
        return new JavetEngine(this, v8Runtime);
    }

    @Override
    public int getActiveEngineCount() {
        return this.engines.length - this.getIdleEngineCount() - this.getReleasedEngineCount();
    }

    @Override
    public JavetEngineConfig getConfig() {
        return this.config;
    }

    @Override
    public IJavetEngine<R> getEngine() throws JavetException {
        long startTime;
        IJavetLogger logger = this.config.getJavetLogger();
        logger.debug("JavetEnginePool.getEngine() begins.");
        JavetEngine<R> engine = null;
        long lastTime = startTime = System.currentTimeMillis();
        int retryCount = 0;
        while (!this.quitting) {
            if (this.semaphore.tryAcquire()) {
                try {
                    Integer index = this.idleEngineIndexList.poll();
                    if (index == null) {
                        index = this.releasedEngineIndexList.poll();
                        if (index != null) {
                            engine = this.createEngine();
                            engine.setIndex(index);
                            this.engines[index.intValue()] = engine;
                            break;
                        }
                    } else {
                        engine = this.engines[index];
                        if (engine != null) break;
                        logger.error("Idle engine cannot be null.");
                        engine = this.createEngine();
                        engine.setIndex(index);
                        this.engines[index.intValue()] = engine;
                        break;
                    }
                    this.semaphore.release();
                }
                catch (Throwable t) {
                    logger.logError(t, "Failed to create a new engine.", new Object[0]);
                }
            }
            if (++retryCount >= this.config.getWaitForEngineMaxRetryCount()) {
                logger.logError("Failed to get an engine after {0} tries in {1}ms.", this.config.getWaitForEngineMaxRetryCount(), Long.toString(System.currentTimeMillis() - startTime));
                throw new JavetException(JavetError.EngineNotAvailable);
            }
            try {
                TimeUnit.MILLISECONDS.sleep(this.config.getWaitForEngineSleepIntervalMillis()[this.random.nextInt(this.config.getWaitForEngineSleepIntervalMillis().length)]);
                long currentTime = System.currentTimeMillis();
                if (currentTime - lastTime < (long)this.config.getWaitForEngineLogIntervalMillis()) continue;
                logger.logWarn("{0}ms passed while waiting for an idle engine.", Long.toString(currentTime - startTime));
                lastTime = currentTime;
            }
            catch (Throwable t) {
                logger.logError(t, "Failed to sleep a while to wait for an idle engine.", new Object[0]);
            }
        }
        ((JavetEngine)Objects.requireNonNull(engine)).setActive(true);
        JavetEngineUsage usage = engine.getUsage();
        usage.increaseUsedCount();
        logger.debug("JavetEnginePool.getEngine() ends.");
        return engine;
    }

    @Override
    public int getIdleEngineCount() {
        return this.idleEngineIndexList.size();
    }

    @Override
    public int getReleasedEngineCount() {
        return this.releasedEngineIndexList.size();
    }

    protected ZonedDateTime getUTCNow() {
        return JavetDateTimeUtils.getUTCNow();
    }

    @Override
    public V8SharedMemoryStatistics getV8SharedMemoryStatistics() {
        return V8Host.getInstance(this.config.getJSRuntimeType()).getV8SharedMemoryStatistics();
    }

    @Override
    public boolean isActive() {
        return this.active;
    }

    @Override
    public boolean isClosed() {
        return !this.active;
    }

    @Override
    public boolean isQuitting() {
        return this.quitting;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public int observe(IV8RuntimeObserver<?> ... observers) {
        int processedCount = 0;
        if (observers.length > 0) {
            Object object = this.internalLock;
            synchronized (object) {
                IJavetLogger logger = this.config.getJavetLogger();
                for (JavetEngine<R> engine : this.engines) {
                    if (engine == null) continue;
                    for (IV8RuntimeObserver<?> observer : observers) {
                        if (((V8Runtime)engine.v8Runtime).isClosed()) continue;
                        try {
                            observer.observe((V8Runtime)engine.v8Runtime);
                        }
                        catch (Throwable t) {
                            logger.error(t.getMessage(), t);
                        }
                        finally {
                            ++processedCount;
                        }
                    }
                }
            }
        }
        return processedCount;
    }

    @Override
    public void releaseEngine(IJavetEngine<R> iJavetEngine) {
        IJavetLogger logger = this.config.getJavetLogger();
        logger.debug("JavetEnginePool.releaseEngine() begins.");
        JavetEngine engine = (JavetEngine)Objects.requireNonNull(iJavetEngine);
        engine.setActive(false);
        if (this.config.isAutoSendGCNotification()) {
            engine.sendGCNotification();
        }
        this.idleEngineIndexList.add(engine.getIndex());
        this.semaphore.release();
        this.wakeUpDaemon();
        logger.debug("JavetEnginePool.releaseEngine() ends.");
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void run() {
        Object object;
        IJavetLogger logger = this.config.getJavetLogger();
        logger.debug("JavetEnginePool.run() begins.");
        while (!this.quitting) {
            object = this.internalLock;
            synchronized (object) {
                int initialIdleEngineCount = this.idleEngineIndexList.size();
                for (int i2 = this.config.getPoolMinSize(); i2 < initialIdleEngineCount; ++i2) {
                    ZonedDateTime resetEngineZonedDateTime;
                    int immediateIdleEngineCount = this.idleEngineIndexList.size();
                    Integer index = this.idleEngineIndexList.poll();
                    if (index == null) break;
                    JavetEngine<R> engine = Objects.requireNonNull(this.engines[index], "The idle engine must not be null.");
                    JavetEngineUsage usage = engine.getUsage();
                    ZonedDateTime expirationZonedDateTime = usage.getLastActiveZonedDatetime().plus(this.config.getPoolIdleTimeoutSeconds(), ChronoUnit.SECONDS);
                    if (immediateIdleEngineCount > this.engines.length || expirationZonedDateTime.isBefore(this.getUTCNow())) {
                        try {
                            engine.close(true);
                            continue;
                        }
                        catch (Throwable t) {
                            logger.logError(t, "Failed to release idle engine.", new Object[0]);
                            continue;
                        }
                        finally {
                            this.engines[index.intValue()] = null;
                            this.releasedEngineIndexList.add(index);
                        }
                    }
                    if (this.config.getResetEngineTimeoutSeconds() > 0 && (resetEngineZonedDateTime = usage.getLastActiveZonedDatetime().plus(this.config.getResetEngineTimeoutSeconds(), ChronoUnit.SECONDS)).isBefore(this.getUTCNow())) {
                        try {
                            logger.debug("JavetEnginePool reset engine begins.");
                            engine.resetContext();
                            logger.debug("JavetEnginePool reset engine ends.");
                        }
                        catch (Throwable t) {
                            logger.logError(t, "Failed to reset idle engine.", new Object[0]);
                        }
                    }
                    this.idleEngineIndexList.add(index);
                }
            }
            object = this.externalLock;
            synchronized (object) {
                try {
                    this.externalLock.wait(this.config.getPoolDaemonCheckIntervalMillis());
                }
                catch (InterruptedException e) {
                    logger.logError(e, "Failed to sleep a while to wait for next round in Javet engine pool daemon.", new Object[0]);
                }
            }
        }
        logger.logDebug("JavetEnginePool daemon is quitting with {0}/{1}/{2} engines.", Integer.toString(this.getActiveEngineCount()), Integer.toString(this.getIdleEngineCount()), Integer.toString(this.engines.length));
        object = this.internalLock;
        synchronized (object) {
            TreeSet<Integer> idleEngineIndexSet = new TreeSet<Integer>(this.idleEngineIndexList);
            TreeSet<Integer> releasedEngineIndexSet = new TreeSet<Integer>(this.releasedEngineIndexList);
            for (int index = 0; index < this.engines.length; ++index) {
                JavetEngine<R> engine = this.engines[index];
                if (engine != null) {
                    try {
                        if (engine.isActive()) {
                            try {
                                ((V8Runtime)engine.getV8Runtime()).terminateExecution();
                            }
                            catch (Throwable t) {
                                logger.logError(t, "Failed to terminate active engine.", new Object[0]);
                            }
                        }
                        engine.close(true);
                    }
                    catch (Throwable t) {
                        logger.logError(t, "Failed to release engine.", new Object[0]);
                    }
                    finally {
                        this.engines[index] = null;
                    }
                }
                if (idleEngineIndexSet.contains(index)) {
                    idleEngineIndexSet.remove(index);
                    this.idleEngineIndexList.remove(index);
                }
                if (releasedEngineIndexSet.contains(index)) continue;
                releasedEngineIndexSet.add(index);
                this.releasedEngineIndexList.add(index);
            }
        }
        logger.debug("JavetEnginePool.run() ends.");
    }

    protected void startDaemon() {
        IJavetLogger logger = this.config.getJavetLogger();
        logger.debug("JavetEnginePool.startDaemon() begins.");
        this.idleEngineIndexList.clear();
        this.releasedEngineIndexList.clear();
        for (int i2 = 0; i2 < this.engines.length; ++i2) {
            this.releasedEngineIndexList.add(i2);
        }
        this.semaphore = new Semaphore(this.engines.length);
        this.quitting = false;
        this.daemonThread = new Thread(this);
        this.daemonThread.setDaemon(true);
        this.daemonThread.setName(JAVET_DAEMON_THREAD_NAME);
        this.daemonThread.start();
        this.active = true;
        logger.debug("JavetEnginePool.startDaemon() ends.");
    }

    protected void stopDaemon() {
        IJavetLogger logger = this.config.getJavetLogger();
        logger.debug("JavetEnginePool.stopDaemon() begins.");
        this.quitting = true;
        try {
            if (this.daemonThread != null) {
                this.daemonThread.join();
            }
        }
        catch (Exception e) {
            logger.logError(e, e.getMessage(), new Object[0]);
        }
        finally {
            this.daemonThread = null;
        }
        this.active = false;
        this.quitting = false;
        this.semaphore = null;
        logger.debug("JavetEnginePool.stopDaemon() ends.");
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void wakeUpDaemon() {
        Object object = this.externalLock;
        synchronized (object) {
            this.externalLock.notify();
        }
    }
}

