/*
 * OrbisGIS is a GIS application dedicated to scientific spatial simulation.
 * This cross-platform GIS is developed at French IRSTV institute and is able to
 * manipulate and create vector and raster spatial information. 
 * 
 * OrbisGIS is distributed under GPL 3 license. It is produced by the "Atelier SIG"
 * team of the IRSTV Institute <http://www.irstv.fr/> CNRS FR 2488.
 * 
 * Copyright (C) 2007-2014 IRSTV (FR CNRS 2488)
 * 
 * This file is part of OrbisGIS.
 * 
 * OrbisGIS is free software: you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any later
 * version.
 * 
 * OrbisGIS is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 * A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * OrbisGIS. If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information, please consult: <http://www.orbisgis.org/>
 * or contact directly:
 * info_at_ orbisgis.org
 */

package org.orbisgis.legend.structure.recode

import org.orbisgis.coremap.renderer.se.parameter.SeParameter
import org.orbisgis.coremap.renderer.se.parameter.string.Recode2String
import org.orbisgis.coremap.renderer.se.parameter.string.StringAttribute
import org.orbisgis.coremap.renderer.se.parameter.string.StringLiteral
import org.orbisgis.coremap.renderer.se.parameter.string.StringParameter
import org.orbisgis.legend.structure.parameter.AbstractAttributeLegend

/**
 * Wrapper for StringLiteral and Recode2String instances : both of them can be recognized as part of a unique value
 * analysis.
 * @author Alexis
 */
class RecodedString extends AbstractAttributeLegend with RecodedLegend {

  var parameter : StringParameter = new StringLiteral

  /**
   * Gets the number of items defined in the inner StringParameter.
   * @return
   */
  def size : Int = {
    parameter match {
      case _ : StringLiteral => 0
      case a : Recode2String => a.getNumMapItem
    }
  }

  /**
   * Builds a new RecodedString using the given parameter.
   * @param param The original StringParameter
   * @throws IllegalArgumentException If param is neither a StringLiteral nor a Recode2String.
   */
  def this(param : StringParameter) = {
    this
    setParameter(param)
  }

  /**
   * Gets the inner parameter.
   **/
  def getParameter : StringParameter = parameter

  /**
   * Sets paramter to s
   * @param s
   * @throws IllegalArgumentException if s is neither a Recode2String nor a StringLiteral
   */
  def setParameter(s : SeParameter) : Unit = s match {
    case a : StringLiteral => 
      parameter = a
      fireTypeChanged()
    case b : Recode2String => 
      parameter = b
      field = getValueReference.getColumnName
      fireTypeChanged()
    case _ => throw new IllegalArgumentException("This class must be built from a  string recode or literal.")
  }

  /**
   * Gets the field used to make the analysis
   * @return
   */
  override def getLookupFieldName : String = field

  /**
   * Gets the field used to make the analysis
   * @return
   */
  override def setLookupFieldName(s: String) = setField(s)

  /**
   * Gets the Double value, if any, associated to {@code key} in the inner {@code
   * Recode}.
   * @param i
   * @return
   */
  def getItemValue(i : Int) : String = parameter match {
    case c : StringLiteral => c.getValue(null)
    case a : Recode2String =>
      Option(a.getMapItemValue(i)).map(_.getValue(null)).getOrElse(null)
  }

  /**
   * Gets the value used when there is no match for a given parameter.
   * @return 
   */
  def getFallbackValue() : String = parameter match {
    case c : StringLiteral => c.getValue(null)
    case a : Recode2String => a.getFallbackValue().getValue(null)
  }

  /**
   * Sets the value that is used when no match is found for a given parameter.
   * @param s
   */
  def setFallbackValue(s : String) = parameter match {
    case cl : StringLiteral => cl.setValue(s)
    case rc : Recode2String => rc.setFallbackValue(new StringLiteral(s))
  }

  /**
   * Gets the Double value, if any, associated to {@code key} in the inner {@code
   * Recode}.
   * @param i
   * @return
   */
  def getItemValue(i : String) : String = parameter match {
    case c : StringLiteral => c.getValue(null)
    case a :Recode2String =>
      val sp : StringParameter = a.getMapItemValue(i)
      if(sp == null) null else sp.getValue(null);
  }
  /**
   * Gets the ith key of the inner {@code Recode}.
   * @param i
   * @return
   */
  def getKey(i : Int) : String = parameter match {
    case c : StringLiteral => ""
    case a : Recode2String => a.getMapItemKey(i)
  }

  /**
   * Sets the ith key of the inner {@code Recode}.
   * @param i
   * @param key
   */
  def setKey(i : Int, key : String) = parameter match {
    case c : StringLiteral => throw new UnsupportedOperationException("A literal does not have a ith key.")
    case a : Recode2String => a.setKey(i, key)
  }

  /**
   * Adds an item in the inner {@code Recode}.
   * @param key
   * @param value
   */
  def addItem(key : String, value : String) = parameter match {
    case c : StringLiteral =>
      val temp = new Recode2String(c,new StringAttribute(field))
      temp.addMapItem(key, new StringLiteral(value))
      setParameter(temp)
    case a : Recode2String => a.addMapItem(key, new StringLiteral(value))
  }

  /**
   * Removes an item from the inner {@code Recode}.
   * @param i
   */
  def  removeItem(i : Int) = parameter match {
    case a : Recode2String =>
      a.removeMapItem(i)
      if(a.getNumMapItem == 0){
        setParameter(new StringLiteral(a.getFallbackValue.getValue(null)))
      }
    case _ =>
  }

  /**
   * Removes an item from the inner {@code Recode}.
   * @param key
   */
  def  removeItem(key : String) = parameter match {
    case a : Recode2String =>
      a.removeMapItem(key)
      if(a.getNumMapItem == 0){
        setParameter(new StringLiteral(a.getFallbackValue.getValue(null)))
      }
    case _ =>
  }
}
