package org.orbroker

import JdbcCloser._

private[orbroker] trait CallStatement extends StaticStatement with ResultSetProducer with GenKeyProducer {

  def call[OP, T](
      token: Token[T],
      session: Session,
      parms: Map[String, _],
      keyHandler: Option[(T) ⇒ Unit],
      receivers: Seq[(T) ⇒ Boolean],
      outParmHandler: Option[(OutParms) ⇒ OP]) = {
    val parsed = statement(parms)
    val started = System.nanoTime
    callback.beforeExecute(token.id, parsed.sql)
    val cs = parsed.prepareCall(session.connection, receivers.size > 0)
    try {
      setFeatures(cs, session)
      val values = setParms(token, cs, parsed.parmDefs, parms)
      val hasResultSet = cs.execute()
      val rowsUpdated = cs.getUpdateCount
      if (hasResultSet) {
        val elements: Iterator[(T) ⇒ Boolean] = receivers.iterator
        var rs = cs.getResultSet
        while (rs != null && elements.hasNext) {
          val next: (T) ⇒ Boolean = elements.next()
          mapResult(token.extractor, rs, next)
          rs = if (cs.getMoreResults) cs.getResultSet else null
        }
      }
      if (rowsUpdated > 0) keyHandler.foreach {
        handleGeneratedKeys(token, _, cs.getGeneratedKeys, rowsUpdated)
      }

      val outParm: OP = outParmHandler match {
        case Some(handler) ⇒ {
          val outParms = new OutParmsImpl(token.id, parsed.parmIdxMap, cs, callback, adapter)
          handler(outParms)
        }
        case None ⇒ null.asInstanceOf[OP]
      }
      callback.afterExecute(token.id, parsed.sql, values, diffTimeInMicros(started))
      (rowsUpdated, outParm)
    } finally {
      cs.checkAndClose(id)
    }
  }

}