package org.orbroker.config.dynamic

import org.orbroker._
import freemarker.template.Configuration

/**
 * FreeMarker support for [[org.orbroker.config.BrokerConfig]].
 * @author Nils Kilden-Pedersen
 */
trait FreeMarkerSupport extends DynamicSupport {

  /** Optional template engine properties. */
  var freemarkerProps: Map[String, String] = Map("number_format" -> "0")

  private lazy val freeMarkerConfig: Configuration = {
    val config = new Configuration
    for ((key, value) ← freemarkerProps)
      config.setSetting(key, value)
    config
  }

  protected abstract override def dynamicStatement(id: Symbol, sql: String, sqlLines: Seq[String]): Option[SQLStatement] = {
    val superDynamic = super.dynamicStatement(id, sql, sqlLines)
    if (superDynamic == None && isFreeMarker(sql)) {
      Some(new FreeMarkerStatement(id, sqlLines, trimSQL, callback, adapter, freeMarkerConfig) with ModifyStatement with QueryStatement)
    } else {
      superDynamic
    }
  }

  private def isFreeMarker(sql: String) = hasFreeMarkerConditionals(sql) || (sql.contains("${") && isFreeMarkerAvailable)

  private def hasFreeMarkerConditionals(sq1: String) =
    ((sq1 contains "#if") && (sq1 contains "/#if")) ||
      ((sq1 contains "#list") && (sq1 contains "/#list"))

  private val isFreeMarkerAvailable = try {
    Class.forName("freemarker.template.Template")
    true
  } catch {
    case _ ⇒ false
  }

}